Ext.define('P5Index.util.TEXTParser', {
    singleton: true,
    
    debugMode: false,  // NEU: Als Property für externe Steuerung
    
    // Konfiguration für Memory-Management
    MAX_LINES_PER_CHUNK: 1000,
    MAX_ITEMS_TOTAL: 10000,

    parseTEXT: function(textContent) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        
        try {
            var sizeInMB = new Blob([textContent]).size / (1024 * 1024);
            console.log('File size:', sizeInMB.toFixed(2), 'MB');
            
            // NEU: Log parse start
            if (logger.isLogging) {
                logger.addLog('Starting TEXT parse (' + sizeInMB.toFixed(2) + ' MB)');
            }
            
            if (sizeInMB > 50) {
                // NEU: Log error
                if (logger.isLogging) {
                    logger.addLog('File too large (max 50MB)', 'ERROR');
                }
                return {
                    success: false,
                    error: 'File too large (max 50MB). Please split into smaller files.'
                };
            }
            
            var lines = textContent.split(/\r?\n/);
            console.log('Total lines:', lines.length);
            
            // Filter leere Zeilen direkt
            lines = lines.filter(function(line) {
                return line.trim().length > 0;
            });
            
            if (lines.length === 0) {
                // NEU: Log warning
                if (logger.isLogging) {
                    logger.addLog('No valid lines found', 'WARNING');
                }
                return { success: false, error: 'No valid lines found' };
            }
            
            // Format erkennen mit Sample
            var format = me.detectFormat(lines.slice(0, 100));
            console.log('Detected format:', format);
            
            // NEU: Log detected format
            if (logger.isLogging) {
                logger.addLog('Detected format: ' + format);
                if (format === 'unknown') {
                    logger.addLog('Could not detect format - no processing', 'WARNING');
                }
            }
            
            // Chunked processing für große Files
            return me.parseInChunks(lines, format);
            
        } catch (error) {
            // NEU: Log error
            if (logger.isLogging) {
                logger.addLog('TEXT parse failed - ' + error.message, 'ERROR');
            }
            console.error('Error parsing text content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    parseInChunks: function(lines, format) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var uniquePaths = {};
        var processedCount = 0;
        
        for (var i = 0; i < lines.length; i += this.MAX_LINES_PER_CHUNK) {
            if (Object.keys(uniquePaths).length >= this.MAX_ITEMS_TOTAL) {
                console.warn('Reached maximum items limit:', this.MAX_ITEMS_TOTAL);
                // NEU: Log limit reached
                if (logger.isLogging) {
                    logger.addLog('Reached maximum items limit: ' + this.MAX_ITEMS_TOTAL, 'WARNING');
                }
                break;
            }
            
            var chunk = lines.slice(i, i + this.MAX_LINES_PER_CHUNK);
            var chunkItems = me.parseChunk(chunk, format);
            
            // Direkt zu uniquePaths hinzufügen
            chunkItems.forEach(function(item) {
                if (!uniquePaths[item.path]) {
                    uniquePaths[item.path] = item;
                }
            });
            
            processedCount += chunk.length;
            
            if (i % (this.MAX_LINES_PER_CHUNK * 10) === 0) {
                console.log('Processed:', processedCount, '/', lines.length, 'lines');
            }
        }
        
        var uniqueItems = Object.values(uniquePaths);
        
        // NEU: Log summary
        if (logger.isLogging) {
            logger.addLog('Found ' + uniqueItems.length + ' unique paths from ' + 
                         processedCount + ' lines');
        }
        
        return {
            success: true,
            mediaItems: uniqueItems,
            totalCount: uniqueItems.length,
            formatType: 'TEXT',
            detectedFormat: format,
            processedLines: processedCount,
            totalLines: lines.length
        };
    },

    detectFormat: function(lines) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        var sampleSize = Math.min(20, lines.length);
        var samples = lines.slice(0, sampleSize);
        
        // NEU: Debug logging
        if (logger.isLogging && debugMode) {
            logger.addLog('Analyzing first ' + sampleSize + ' lines for format detection');
            if (lines.length > 0) {
                var firstLine = lines[0].substring(0, 80);
                if (lines[0].length > 80) firstLine += '...';
                logger.addLog('First line: "' + firstLine + '"');
            }
        }
        
        var pathOnlyCount = 0;
        var logCount = 0;
        
        samples.forEach(function(line) {
            // Pathlist: Zeile sieht wie kompletter Pfad aus
            if (me.looksLikeCompletePath(line)) {
                pathOnlyCount++;
            }
            
            // Log: Typische Log-Zeilen
            if (me.isLogLine(line)) {
                logCount++;
            }
        });
        
        var total = samples.length;
        var format = 'unknown';
        
        if (pathOnlyCount / total > 0.9) format = 'pathlist';
        else if (logCount / total > 0.5) format = 'log';
        
        // NEU: Debug log detection result
        if (logger.isLogging && debugMode) {
            logger.addLog('Format detection: ' + pathOnlyCount + ' paths, ' + 
                         logCount + ' log lines → ' + format);
        }
        
        return format;
    },

    looksLikeCompletePath: function(line) {
        line = line.trim();
        // Beginnt mit Pfad-Pattern
        var startsWithPath = line.match(/^[\/\\]/) || line.match(/^[A-Za-z]:[\/\\]/);
        // Keine typischen Delimiter (Tab, mehrere Kommas, etc.)
        var noDelimiters = !line.includes('\t') && 
                          !line.includes(';') &&
                          (line.match(/,/g) || []).length <= 2;
        
        return startsWithPath && line.length > 5 && noDelimiters;
    },
 
    parseChunk: function(lines, format) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        // NEU: Debug log chunk processing
        if (logger.isLogging && debugMode) {
            logger.addLog('Processing chunk of ' + lines.length + ' lines as ' + format);
        }
        
        switch(format) {
            case 'pathlist': 
                return this.parsePathListChunk(lines);
            case 'log': 
                return this.parseLogChunk(lines);
            case 'unknown':
            default:
                return []; // Keine generische Verarbeitung
        }
    }, 

    parsePathListChunk: function(lines) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        var mediaItems = [];
        var skipped = 0;  // NEU
        
        for (var i = 0; i < lines.length; i++) {
            var line = lines[i].trim();
            if (line && me.isPathLike(line)) {
                var item = me.createMediaItem(line, 'pathList');
                if (item) {
                    mediaItems.push(item);
                    // NEU: Debug log first few items
                    if (logger.isLogging && debugMode && mediaItems.length <= 3) {
                        logger.addLog('Found path: "' + item.fileName + '"');
                    }
                }
            } else if (line) {
                skipped++;
            }
        }
        
        // NEU: Debug log skipped
        if (logger.isLogging && debugMode && skipped > 0) {
            logger.addLog('Skipped ' + skipped + ' non-path lines in chunk');
        }
        
        return mediaItems;
    },

    parseLogChunk: function(lines) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        var mediaItems = [];
        var pathPattern = /[^\s]+[\/\\][^\s]*/g;
        var pathsFound = 0;  // NEU
        
        for (var i = 0; i < lines.length; i++) {
            var matches = lines[i].match(pathPattern);
            if (matches) {
                for (var j = 0; j < matches.length; j++) {
                    if (me.isPathLike(matches[j])) {
                        var item = me.createMediaItem(matches[j], 'log');
                        if (item) {
                            mediaItems.push(item);
                            pathsFound++;
                            // NEU: Debug log first few
                            if (logger.isLogging && debugMode && pathsFound <= 3) {
                                logger.addLog('Extracted from log: "' + item.fileName + '"');
                            }
                        }
                    }
                }
            }
        }
        
        // NEU: Debug summary
        if (logger.isLogging && debugMode && pathsFound > 3) {
            logger.addLog('Total paths extracted from log chunk: ' + pathsFound);
        }
        
        return mediaItems;
    },

    createMediaItem: function(originalPath, type) {
        try {
            var unixPath = this.normalizeToUnixPath(originalPath);
            var fileName = this.getFileName(unixPath);
            
            if (!fileName || fileName.length === 0) {
                return null;
            }
            
            return {
                type: type,
                fileName: fileName,
                path: unixPath,
                originalPath: originalPath,
                usedDurationTC: '',
                availableDurationTC: ''
            };
        } catch (e) {
            console.warn('Error creating media item for:', originalPath, e);
            return null;
        }
    },

    isLogLine: function(line) {
        return /^\d{4}-\d{2}-\d{2}/.test(line) || 
               /^\[\d{4}-\d{2}-\d{2}/.test(line) ||
               /^\d{2}:\d{2}:\d{2}/.test(line);
    },

    isPathLike: function(path) {
        return (path.includes('/') || path.includes('\\')) && 
               path.length > 3 &&
               !path.includes('http://') && 
               !path.includes('https://');
    },

    normalizeToUnixPath: function(path) {
        // Handle Windows-style escaped characters
        path = path.replace(/\^([0-9a-fA-F]{2})/g, function(match, hex) {
            return String.fromCharCode(parseInt(hex, 16));
        });
        
        // Standard Unix path normalization
        var unixPath = path.replace(/\\/g, '/');
        unixPath = unixPath.replace(/^([A-Za-z]):/, function(match, driveLetter) {
            return '/' + driveLetter.toLowerCase() + '/';
        });
        unixPath = unixPath.replace(/\/+/g, '/');
        
        return unixPath;
    },

    getFileName: function(path) {
        return path.split('/').pop().split('\\').pop();
    }
});