Ext.define('Dashboard.view.widget.ArchiveIndexUsage', {
    extend: 'Ext.panel.Panel',
    xtype: 'archive-index-usage-widget',
    
    requires: [
        'Dashboard.view.widget.ArchiveIndexUsageModel',
        'Dashboard.view.widget.ArchiveIndexUsageController'
    ],
    
    controller: 'archive-index-usage',
    viewModel: {
        type: 'archive-index-usage'
    },
    
    layout: 'fit',
    
    initComponent: function() {
        var me = this;
        
        // Container für die Charts mit responsive Grid-Layout
        me.chartContainer = Ext.create('Ext.container.Container', {
            layout: {
                type: 'column',
                align: 'stretch'
            },
            scrollable: {
                y: true,
                x: false
            },
            items: [{
                xtype: 'component',
                columnWidth: 1.0,
                html: '<div style="padding: 20px; text-align: center; color: #666;">Loading charts...</div>',
                height: 60
            }],
            style: {
                backgroundColor: '#f5f5f5',
                padding: '10px'
            }
        });
        
        me.items = [me.chartContainer];
        me.callParent();
    },
    
    // Zugriff auf den Chart-Container
    getChartContainer: function() {
        return this.chartContainer;
    },
    
    calculateResponsiveLayout: function(chartCount) {
        var containerWidth = this.getWidth() || 800;
        var minChartWidth = 300;  // Minimale Chart-Breite für Lesbarkeit
        
        // Maximale Anzahl Spalten basierend auf Container-Breite und minimaler Chart-Breite
        var maxPossibleColumns = Math.floor(containerWidth / minChartWidth);
        
        // Tatsächliche Spaltenanzahl: minimum aus verfügbaren Charts und möglichen Spalten
        var columns = Math.min(chartCount, maxPossibleColumns);
        
        // Fallback: mindestens 1 Spalte
        if (columns === 0) {
            columns = 1;
        }
        
        var columnWidth = 1.0 / columns;
        var actualChartWidth = Math.floor(containerWidth / columns) - 10; // 10px für Padding
        
        return {
            columns: columns,
            columnWidth: columnWidth,
            chartWidth: actualChartWidth
        };
    },
    
    calculateOptimalYAxisRange: function(data, fieldName) {
        if (!data || data.length === 0) return { min: 0, max: 100, isZoomed: false };
        
        var values = data.map(function(item) { return item[fieldName]; });
        var minValue = Math.min.apply(Math, values);
        var maxValue = Math.max.apply(Math, values);
        var range = maxValue - minValue;
        
        // Wenn die Variation weniger als 5% des Durchschnittswerts ist, zoomen
        var avgValue = (minValue + maxValue) / 2;
        var variationPercent = avgValue > 0 ? (range / avgValue) * 100 : 0;
        
        if (variationPercent < 5 && range > 0) {
            // Zoom-Modus: Y-Achse beginnt nahe dem Minimalwert
            var padding = Math.max(range * 0.2, 1); // Mindestens 20% Padding oder 1 GB
            return {
                min: Math.max(0, minValue - padding),
                max: maxValue + padding,
                isZoomed: true
            };
        }
        
        // Normal-Modus: Y-Achse beginnt bei 0
        return {
            min: 0,
            max: maxValue * 1.1,
            isZoomed: false
        };
    },
    
    createTimeSeriesChart: function(queryName, description, data, chartIndex, totalCharts) {
        var controller = this.getController();
        var chartColor = controller.getColorForQuery(queryName);
        
        var layout = this.calculateResponsiveLayout(totalCharts || 1);
        var yAxisRange = this.calculateOptimalYAxisRange(data, 'totalSizeKiB');
        
        return {
            xtype: 'panel',
            reference: 'chart_' + queryName,
            queryName: queryName,
            
            frame: false,
            border: true,
            bodyStyle: {
                background: '#fafafa',
                border: '0px solid #e0e0e0'
            },
            style: {
                borderRadius: '8px',
                boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
                overflow: 'hidden'
            },
            
            columnWidth: layout.columnWidth,
            width: layout.chartWidth,
            height: 150,
            margin: '4 4 4 4',
            
            layout: {
                type: 'vbox',
                align: 'stretch'  // Wichtig!
            },
            
            items: [{
                xtype: 'component',
                html: description,
                style: {
                    fontSize: '13px',
                    fontWeight: 'bold',
                    color: '#444',
                    padding: '8px 12px 4px 12px',
                    background: '#fafafa'
                }
            }, {
                xtype: 'cartesian',
                reference: 'cartesian_' + queryName,
                flex: 1,
                insetPadding: '5 15 5 5',
                animation: false,
                border: 0,
                bodyStyle: {
                    background: 'transparent'
                },
                store: {
                    fields: ['date', 'dateShort', 'totalSizeKiB', 'totalSizeGiB', 'changeFromPrevious'],
                    data: data
                },
                legend: {
                    type: 'dom',
                    docked: 'right',
                    hidden: true
                },
                axes: [{
                    type: 'numeric',
                    position: 'left',
                    grid: {
                        stroke: '#f0f0f0'
                    },
                    minimum: yAxisRange.min,
                    maximum: yAxisRange.max,
                    label: {
                        fill: '#666',
                        fontSize: 10
                    },
                    renderer: function(axis, label) {
                        return CalcSize(label, 0);
                    }
                }, {
                    type: 'category',
                    position: 'bottom',
                    grid: {
                        stroke: '#f8f8f8'
                    },
                    label: {
                        fontSize: 10,
                        fill: '#666'
                    }
                }],
                series: [{
                    type: 'bar',
                    xField: 'dateShort',
                    yField: 'totalSizeKiB',
                    style: {
                        maxBarWidth: 28,
                        opacity: 0.9,
                        strokeWidth: 0
                    },
                    colors: [chartColor],
                    tooltip: {
                        trackMouse: true,
                        renderer: function(tooltip, record) {
                            var date = Ext.util.Format.date(record.get('date'), 'M d, Y');
                            var totalSize = CalcSize(record.get('totalSizeKiB'));
                            var change = record.get('changeFromPrevious');
                            
                            var changeFormatted = '';
                            if (change && change !== 0) {
                                var changeSign = change >= 0 ? '+' : '';
                                var changeSize = CalcSize(Math.abs(change));
                                changeFormatted = '<br>Daily Change: ' + changeSign + changeSize;
                            }
                            
                            var detailHint = yAxisRange.isZoomed ? '<br><em>Detail view active</em>' : '';
                            
                            tooltip.setHtml(
                                '<strong>' + date + '</strong><br>' +
                                'Total Size: ' + totalSize + 
                                changeFormatted +
                                detailHint
                            );
                        }
                    }
                }]
            }]
        };
    },
    
    dockedItems: [{
        xtype: 'toolbar',
        dock: 'top',
        items: [{
            xtype: 'image',
            src: 'img8/query_job_32.png',
            width: 24,
            height: 24
        }, '->', {
            xtype: 'segmentedbutton',
            reference: 'periodToggle_ref',
            stateful: true,
            stateId: 'archive-index-usage-period',
            stateEvents : ['toggle', 'pressed'],
            allowMultiple: false,
            defaults: {
                width: 50
            },
            items: [{
                text: '1W',
                value: 7,
            }, {
                text: '2W', 
                value: 14
            }, {
                text: '3W',
                value: 21
            }, {
                text: '4W',
                value: 28
            }],
            listeners: {
                toggle: 'onPeriodChange'
            }
        }, '-', {
            xtype: 'button',
            reference: 'querySelector_ref',
            text: $$('Select Queries'),
            width: 150,
            menu: {
                plain: true,
                items: [{
                    xtype: 'checkboxgroup',
                    columns: 1,
                    vertical: true,
                    reference: 'queryCheckboxGroup_ref',
                    listeners: {
                        change: 'onQuerySelectionChange'
                    }
                }, '-', {
                    text: $$('Refresh Data'),
                    iconCls: 'x-fa fa-refresh',
                    handler: 'onRefreshData'
                }]
            }
        }]
    }]
});