Ext.define('P5Index.util.FCPXMLParser', {
    singleton: true,
    
    debugMode: false,  // Als Property für externe Steuerung

    /**
     * Parse FCPXML file and extract media references
     * @param {string} fcpxmlContent - The FCPXML content as string
     * @returns {Object} Object containing extracted media items
     */
    parseFCPXML: function(fcpxmlContent) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        try {
            // Log start
            if (logger.isLogging) {
                logger.addLog('Starting FCPXML parse');
            }
            
            console.log('Parsing FCPXML...');
            
            // Parse XML
            var parser = new DOMParser();
            var xmlDoc = parser.parseFromString(fcpxmlContent, 'text/xml');
            
            // Check for XML parsing errors
            var parseError = xmlDoc.getElementsByTagName('parsererror');
            if (parseError.length > 0) {
                if (logger.isLogging) {
                    logger.addLog('FCPXML XML parsing failed', 'ERROR');
                }
                throw new Error('XML parsing failed: ' + parseError[0].textContent);
            }
            
            var mediaItems = [];
            var uniqueFiles = {};
            var assetRegistry = {}; // Map asset IDs to asset data
            var formatRegistry = {}; // Map format IDs to format data
            
            // Get FCPXML version
            var fcpxmlElement = xmlDoc.getElementsByTagName('fcpxml')[0];
            var version = fcpxmlElement ? fcpxmlElement.getAttribute('version') : 'unknown';
            
            if (logger.isLogging) {
                logger.addLog('FCPXML version: ' + version);
            }
            
            console.log('FCPXML version:', version);
            
            // Step 1: Parse formats for FPS detection
            this.parseFormats(xmlDoc, formatRegistry);
            
            if (logger.isLogging && Object.keys(formatRegistry).length > 0) {
                logger.addLog('Found ' + Object.keys(formatRegistry).length + ' format definitions');
            }
            
            // Step 2: Parse all assets from resources
            this.parseAssets(xmlDoc, assetRegistry, formatRegistry, version);
            console.log('Found', Object.keys(assetRegistry).length, 'assets in resources');
            
            // Step 3: Get sequence info
            var sequences = xmlDoc.getElementsByTagName('sequence');
            var sequenceNames = [];
            for (var i = 0; i < sequences.length; i++) {
                var seqName = sequences[i].getAttribute('name');
                if (seqName) sequenceNames.push(seqName);
            }
            
            if (logger.isLogging) {
                logger.addLog('Structure: ' + Object.keys(assetRegistry).length + ' assets, ' +
                             sequences.length + ' sequence' + (sequences.length !== 1 ? 's' : ''));
                if (sequenceNames.length > 0 && debugMode) {
                    logger.addLog('Sequences: ' + sequenceNames.join(', '));
                }
            }
            
            // Step 4: Parse timeline usage from sequences
            this.parseSequences(xmlDoc, assetRegistry, uniqueFiles);
            
            // Step 5: Add unused assets (available but not used in timeline)
            this.addUnusedAssets(assetRegistry, uniqueFiles);
            
            // Convert to array
            mediaItems = Object.values(uniqueFiles);
            
            // Log summary
            if (logger.isLogging) {
                var usedCount = 0;
                var unusedCount = 0;
                var mediaKindCounts = {};
                
                mediaItems.forEach(function(item) {
                    if (item.source === 'unused_asset') {
                        unusedCount++;
                    } else {
                        usedCount++;
                    }
                    
                    var kind = item.mediaKind || 'original-media';
                    mediaKindCounts[kind] = (mediaKindCounts[kind] || 0) + 1;
                });
                
                var summary = [];
                if (usedCount > 0) summary.push('used: ' + usedCount);
                if (unusedCount > 0) summary.push('unused: ' + unusedCount);
                
                logger.addLog('Found ' + mediaItems.length + ' items (' + summary.join(', ') + ')');
                
                // Media kind breakdown if interesting
                if (Object.keys(mediaKindCounts).length > 1) {
                    var kindSummary = Object.keys(mediaKindCounts).map(function(kind) {
                        return kind + ': ' + mediaKindCounts[kind];
                    }).join(', ');
                    logger.addLog('Media kinds: ' + kindSummary);
                }
            }
            
            console.log('FCPXML parsing completed:', mediaItems.length, 'unique media items found');
            
            return {
                success: true,
                mediaItems: mediaItems,
                totalCount: mediaItems.length,
                formatType: 'FCPXML',
                detectedFormat: 'Final Cut Pro XML (FCPXML v' + version + ')',
                metadata: {
                    version: version,
                    assetCount: Object.keys(assetRegistry).length
                }
            };
            
        } catch (error) {
            if (logger.isLogging) {
                logger.addLog('FCPXML parse failed - ' + error.message, 'ERROR');
            }
            console.error('Error parsing FCPXML content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    /**
     * Parse formats from resources section for FPS detection
     */
    parseFormats: function(xmlDoc, formatRegistry) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var formats = xmlDoc.getElementsByTagName('format');
        
        for (var i = 0; i < formats.length; i++) {
            var format = formats[i];
            var id = format.getAttribute('id');
            var frameDuration = format.getAttribute('frameDuration');
            var name = format.getAttribute('name');
            
            if (id) {
                var fps = this.calculateFPSFromDuration(frameDuration);
                
                formatRegistry[id] = {
                    id: id,
                    name: name,
                    frameDuration: frameDuration,
                    fps: fps,
                    width: format.getAttribute('width'),
                    height: format.getAttribute('height')
                };
                
                if (logger.isLogging && debugMode) {
                    logger.addLog('Format ' + id + ': ' + (name || 'unnamed') + ' @ ' + fps + 'fps');
                }
            }
        }
    },

    /**
     * Calculate FPS from frameDuration (e.g., "100/2500s" = 25fps)
     */
    calculateFPSFromDuration: function(frameDuration) {
        if (!frameDuration) return 25; // Default fallback
        
        try {
            // Handle formats like "100/2500s" or "1001/30000s"
            var match = frameDuration.match(/^(\d+)\/(\d+)s?$/);
            if (match) {
                var numerator = parseInt(match[1]);
                var denominator = parseInt(match[2]);
                var duration = numerator / denominator; // Duration per frame in seconds
                return Math.round(1 / duration); // FPS = 1 / duration
            }
            
            return 25; // Fallback
        } catch (e) {
            console.warn('Could not calculate FPS from frameDuration:', frameDuration);
            return 25;
        }
    },

    /**
     * Parse all media-rep elements regardless of location (robust approach)
     */
    parseAssets: function(xmlDoc, assetRegistry, formatRegistry, version) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Find ALL media-rep elements in the document
        var allMediaReps = xmlDoc.getElementsByTagName('media-rep');
        
        if (logger.isLogging && debugMode) {
            logger.addLog('Found ' + allMediaReps.length + ' media-rep elements');
        }
        
        console.log('Found', allMediaReps.length, 'media-rep elements');
        
        var assetCount = 0;
        var mediaCount = 0;
        var skippedCount = 0;
        
        for (var i = 0; i < allMediaReps.length; i++) {
            var mediaRep = allMediaReps[i];
            var parentElement = mediaRep.parentNode;
            var parentType = parentElement.tagName;
            
            if (logger.isLogging && debugMode) {
                logger.addLog('Processing media-rep in parent: ' + parentType);
            }
            
            console.log('Processing media-rep in parent:', parentType);
            
            if (parentType === 'asset') {
                // Real media file - primary target for archive recovery
                var assetData = this.extractAssetDataFromMediaRep(mediaRep, parentElement, formatRegistry, version, 'asset');
                if (assetData && assetData.id) {
                    assetRegistry[assetData.id] = assetData;
                    assetCount++;
                    
                    if (logger.isLogging && debugMode) {
                        logger.addLog('Asset: "' + assetData.fileName + '" (ID: ' + assetData.id + ')');
                    }
                    
                    console.log('Added asset:', assetData.fileName, '(from asset)');
                }
            } else if (parentType === 'media') {
                // Compound clip, multicam, sync clip - secondary interest
                var mediaData = this.extractAssetDataFromMediaRep(mediaRep, parentElement, formatRegistry, version, 'media');
                if (mediaData && mediaData.id) {
                    assetRegistry[mediaData.id] = mediaData;
                    mediaCount++;
                    
                    if (logger.isLogging && debugMode) {
                        logger.addLog('Media: "' + mediaData.fileName + '" (compound/multicam)');
                    }
                    
                    console.log('Added media:', mediaData.fileName, '(from media element)');
                }
            } else {
                // Other parent types (effect, etc.) - usually not relevant for archive
                skippedCount++;
                
                if (logger.isLogging && debugMode) {
                    logger.addLog('Skipped media-rep in ' + parentType, 'WARNING');
                }
                
                console.log('Skipping media-rep in parent type:', parentType);
            }
        }
        
        if (logger.isLogging && debugMode) {
            var summary = [];
            if (assetCount > 0) summary.push(assetCount + ' assets');
            if (mediaCount > 0) summary.push(mediaCount + ' compound/multicam');
            if (skippedCount > 0) summary.push(skippedCount + ' skipped');
            logger.addLog('Asset processing: ' + summary.join(', '));
        }
    },

    /**
     * Extract data from media-rep and its parent element
     */
    extractAssetDataFromMediaRep: function(mediaRep, parentElement, formatRegistry, version, parentType) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Get parent element attributes
        var id = parentElement.getAttribute('id');
        var name = parentElement.getAttribute('name');
        var start = parentElement.getAttribute('start');
        var duration = parentElement.getAttribute('duration');
        var hasVideo = parentElement.getAttribute('hasVideo') === '1';
        var hasAudio = parentElement.getAttribute('hasAudio') === '1';
        var formatRef = parentElement.getAttribute('format');
        
        // Get FPS from format registry
        var fps = 25; // Default
        if (formatRef && formatRegistry[formatRef]) {
            fps = formatRegistry[formatRef].fps;
        }
        
        // Extract media location from media-rep element
        var mediaInfo = this.extractMediaLocationFromMediaRep(mediaRep, version);
        var filePath = mediaInfo.filePath;
        var fileName = mediaInfo.fileName;
        
        // Fallback to parent name if no filename found
        if (!fileName && name) {
            fileName = this.extractFileName(name) || name;
            
            if (logger.isLogging && debugMode) {
                logger.addLog('Using parent name as filename: "' + fileName + '"', 'WARNING');
            }
        }
        
        if (!fileName) {
            if (logger.isLogging && debugMode) {
                logger.addLog('Skipping media-rep without usable filename (ID: ' + id + ')', 'WARNING');
            }
            console.log('Skipping media-rep without usable filename');
            return null;
        }
        
        return {
            id: id,
            name: name,
            fileName: fileName,
            path: filePath,
            start: start,
            duration: duration,
            availableDurationTC: this.convertRationalToTimecode(duration, fps),
            hasVideo: hasVideo,
            hasAudio: hasAudio,
            formatRef: formatRef,
            fps: fps,
            used: false, // Will be set to true if used in timeline
            mediaKind: mediaInfo.kind,
            parentType: parentType // 'asset' or 'media' for categorization
        };
    },

    /**
     * Extract media location directly from media-rep element
     */
    extractMediaLocationFromMediaRep: function(mediaRep, version) {
        var filePath = '';
        var fileName = '';
        var kind = mediaRep.getAttribute('kind') || 'original-media';
        
        var src = mediaRep.getAttribute('src');
        var suggestedFilename = mediaRep.getAttribute('suggestedFilename');
        
        if (src) {
            filePath = this.decodeFileUrl(src);
            fileName = this.extractFileName(filePath);
            
            // Use suggestedFilename if no filename in URL
            if (!fileName && suggestedFilename) {
                fileName = suggestedFilename;
                // Add file extension from URL if possible
                var urlExt = this.extractFileExtension(src);
                if (urlExt && !suggestedFilename.includes('.')) {
                    fileName = suggestedFilename + '.' + urlExt;
                }
            }
        }
        
        return {
            filePath: filePath,
            fileName: fileName,
            kind: kind
        };
    },

    /**
     * Parse sequences to find timeline usage
     */
    parseSequences: function(xmlDoc, assetRegistry, uniqueFiles) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var sequences = xmlDoc.getElementsByTagName('sequence');
        
        for (var i = 0; i < sequences.length; i++) {
            var sequence = sequences[i];
            var seqName = sequence.getAttribute('name');
            
            if (logger.isLogging && debugMode && seqName) {
                logger.addLog('Processing sequence: "' + seqName + '"');
            }
            
            this.parseSequenceClips(sequence, assetRegistry, uniqueFiles);
        }
    },

    /**
     * Parse clips from a sequence
     */
    parseSequenceClips: function(sequenceElement, assetRegistry, uniqueFiles) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Parse both <asset-clip> and <clip> elements
        var assetClips = sequenceElement.getElementsByTagName('asset-clip');
        var clips = sequenceElement.getElementsByTagName('clip');
        
        var processedCount = 0;
        var unreferencedCount = 0;
        
        // Process asset-clips
        for (var i = 0; i < assetClips.length; i++) {
            var result = this.processClipElement(assetClips[i], assetRegistry, uniqueFiles, 'asset-clip');
            if (result) {
                processedCount++;
            } else {
                unreferencedCount++;
            }
        }
        
        // Process clips
        for (var j = 0; j < clips.length; j++) {
            var result = this.processClipElement(clips[j], assetRegistry, uniqueFiles, 'clip');
            if (result) {
                processedCount++;
            } else {
                unreferencedCount++;
            }
        }
        
        if (logger.isLogging && debugMode) {
            var totalClips = assetClips.length + clips.length;
            if (totalClips > 0) {
                logger.addLog('Sequence clips: ' + processedCount + ' processed, ' + 
                            unreferencedCount + ' without asset reference');
            }
        }
    },

    /**
     * Process individual clip element
     */
    processClipElement: function(clipElement, assetRegistry, uniqueFiles, elementType) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var ref = clipElement.getAttribute('ref');
        var name = clipElement.getAttribute('name');
        var start = clipElement.getAttribute('start');
        var duration = clipElement.getAttribute('duration');
        var offset = clipElement.getAttribute('offset');
        
        var assetData = null;
        var fileName = null;
        var filePath = '';
        var fps = 25; // Default
        
        if (ref && assetRegistry[ref]) {
            // Referenced asset
            assetData = assetRegistry[ref];
            fileName = assetData.fileName;
            filePath = assetData.filePath;
            fps = assetData.fps || 25;
            assetData.used = true;
            
            if (logger.isLogging && debugMode) {
                logger.addLog('Clip references: "' + fileName + '" (ref: ' + ref + ')');
            }
        } else if (ref && !assetRegistry[ref]) {
            if (logger.isLogging && debugMode) {
                logger.addLog('Clip with unknown ref: ' + ref + ' (name: "' + name + '")', 'WARNING');
            }
        } else if (name) {
            // Direct name reference - uncommon in FCPXML
            // fileName = this.extractFileName(name) || name;
            if (logger.isLogging && debugMode) {
                logger.addLog('Clip without ref: "' + name + '" - skipped', 'WARNING');
            }
        }
        
        if (!fileName) {
            return false;
        }
        
        var key = fileName.toLowerCase();
        
        if (!uniqueFiles[key]) {
            uniqueFiles[key] = {
                fileName: fileName,
                path: filePath,
                usedDurationTC: this.convertRationalToTimecode(duration, fps),
                availableDurationTC: assetData ? assetData.availableDurationTC : '',
                source: elementType,
                mediaKind: assetData ? assetData.mediaKind : 'original-media',
                clips: []
            };
        }
        
        // Add clip usage information
        uniqueFiles[key].clips.push({
            elementType: elementType,
            start: start,
            duration: duration,
            offset: offset,
            startTC: this.convertRationalToTimecode(start, fps),
            durationTC: this.convertRationalToTimecode(duration, fps),
            ref: ref
        });
        
        // Update used duration to longest clip if multiple
        var currentDurationTC = this.convertRationalToTimecode(duration, fps);
        if (currentDurationTC && (!uniqueFiles[key].usedDurationTC || 
            this.compareTimecodes(currentDurationTC, uniqueFiles[key].usedDurationTC, fps) > 0)) {
            uniqueFiles[key].usedDurationTC = currentDurationTC;
        }
        
        return true;
    },

    /**
     * Add unused assets to the file list
     */
    addUnusedAssets: function(assetRegistry, uniqueFiles) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var unusedCount = 0;
        
        for (var assetId in assetRegistry) {
            var asset = assetRegistry[assetId];
            
            if (!asset.used && asset.fileName) {
                unusedCount++;
                var key = asset.fileName.toLowerCase();
                
                if (!uniqueFiles[key]) {
                    if (logger.isLogging && debugMode) {
                        logger.addLog('Adding unused asset: "' + asset.fileName + '"');
                    }
                    
                    uniqueFiles[key] = {
                        fileName: asset.fileName,
                        path: asset.filePath,
                        usedDurationTC: '', // Not used in timeline
                        availableDurationTC: asset.availableDurationTC,
                        source: 'unused_asset',
                        mediaKind: asset.mediaKind || 'original-media',
                        clips: []
                    };
                }
            }
        }
        
        if (logger.isLogging && debugMode && unusedCount > 0) {
            logger.addLog('Added ' + unusedCount + ' unused assets');
        }
    },

    /**
     * Convert rational duration to timecode with FPS support
     */
    convertRationalToTimecode: function(rationalStr, fps) {
        if (!rationalStr) return '';
        
        fps = fps || 25; // Default fallback
        
        try {
            // Handle formats like "1279/5s" or "3600/1s"
            var match = rationalStr.match(/^(\d+)\/(\d+)s?$/);
            if (match) {
                var numerator = parseInt(match[1]);
                var denominator = parseInt(match[2]);
                var seconds = numerator / denominator;
                
                return this.secondsToTimecode(seconds, fps);
            }
            
            // Handle simple seconds like "60s"
            var secondsMatch = rationalStr.match(/^(\d+)s$/);
            if (secondsMatch) {
                var secs = parseInt(secondsMatch[1]);
                return this.secondsToTimecode(secs, fps);
            }
            
            return rationalStr; // Return as-is if can't parse
            
        } catch (e) {
            console.warn('Could not convert rational duration:', rationalStr);
            return rationalStr;
        }
    },

    /**
     * Convert seconds to HH:MM:SS:FF timecode with FPS support
     */
    secondsToTimecode: function(totalSeconds, fps) {
        fps = fps || 25; // Default frame rate
        
        var hours = Math.floor(totalSeconds / 3600);
        var minutes = Math.floor((totalSeconds % 3600) / 60);
        var seconds = Math.floor(totalSeconds % 60);
        var frames = Math.floor((totalSeconds % 1) * fps);
        
        return this.padZero(hours) + ':' + 
               this.padZero(minutes) + ':' + 
               this.padZero(seconds) + ':' + 
               this.padZero(frames);
    },

    /**
     * Decode file:// URL and extract clean path (improved)
     */
    decodeFileUrl: function(fileUrl) {
        if (!fileUrl) return '';
        
        try {
            // Handle different URL schemes
            if (fileUrl.startsWith('file://')) {
                // File URL: file:///Volumes/Media/file.mov
                var path = fileUrl.replace(/^file:\/\//, '');
                // Keep leading slash for absolute paths
                if (!path.startsWith('/')) {
                    path = '/' + path;
                }
                return decodeURIComponent(path);
            } else if (fileUrl.startsWith('http://') || fileUrl.startsWith('https://')) {
                // Remote URL: keep as-is
                return fileUrl;
            } else if (fileUrl.startsWith('./') || fileUrl.startsWith('../')) {
                // Relative URL: keep as-is for now
                return fileUrl;
            } else {
                // Assume it's already a clean path
                return decodeURIComponent(fileUrl);
            }
        } catch (e) {
            console.warn('Could not decode file URL:', fileUrl);
            return fileUrl;
        }
    },

    /**
     * Extract file extension from URL or filename
     */
    extractFileExtension: function(url) {
        if (!url) return '';
        
        try {
            // Remove query parameters and fragments
            var cleanUrl = url.split('?')[0].split('#')[0];
            var fileName = cleanUrl.split('/').pop().split('\\').pop();
            var lastDot = fileName.lastIndexOf('.');
            
            if (lastDot > 0 && lastDot < fileName.length - 1) {
                return fileName.substring(lastDot + 1).toLowerCase();
            }
        } catch (e) {
            console.warn('Could not extract file extension from:', url);
        }
        
        return '';
    },

    /**
     * Compare two timecode strings with FPS support
     */
    compareTimecodes: function(tc1, tc2, fps) {
        // Convert to seconds for comparison
        var seconds1 = this.timecodeToSeconds(tc1, fps);
        var seconds2 = this.timecodeToSeconds(tc2, fps);
        
        return seconds1 - seconds2;
    },

    /**
     * Convert timecode to seconds with FPS support
     */
    timecodeToSeconds: function(timecode, fps) {
        if (!timecode) return 0;
        
        fps = fps || 25; // Default
        
        var parts = timecode.split(':');
        if (parts.length === 4) {
            var hours = parseInt(parts[0]) || 0;
            var minutes = parseInt(parts[1]) || 0;
            var seconds = parseInt(parts[2]) || 0;
            var frames = parseInt(parts[3]) || 0;
            
            return hours * 3600 + minutes * 60 + seconds + (frames / fps);
        }
        
        return 0;
    },

    /**
     * Extract filename from path
     */
    extractFileName: function(path) {
        if (!path) return '';
        
        // Handle both forward and backward slashes
        var fileName = path.split('/').pop().split('\\').pop();
        return fileName;
    },

    /**
     * Pad number with leading zero
     */
    padZero: function(num) {
        return (num < 10 ? '0' : '') + num;
    }
});