/**
 * Dashboard.js - Core Dashboard Component
 * 
 * Extends Ext.dashboard.Dashboard with customized initialization and widget management.
 * 
 * Key components:
 * @method initComponent - Initializes dashboard configuration
 *   - Sets column count from form or defaults to 1
 *   - Loads widget configuration from form or uses defaults
 *   - Distributes widgets across columns using modulo operator
 * 
 * @property parts - Defines all available widgets
 *   - Each widget definition includes title, header configuration, and component
 *   - Standard header padding (2 5 2 5) for consistent appearance
 * 
 * @property tbar - Toolbar with dashboard controls
 *   - Column count selector (RadioGroup)
 *   - Widget visibility toggles (CheckboxGroup)
 *   - State is managed through beforerender listeners
 * 
 * State Management:
 * - Uses ExtJS state management (stateId: 'dashboard-layout')
 * - Server syncs state via session variables
 * - Column count and widget visibility persist across reloads
 * 
 * Server Integration:
 * - Form values determine initial state
 * - Ajax calls sync changes to server
 * - Session manages persistence
 */

Ext.define('Dashboard.view.Dashboard', {
    extend: 'Ext.dashboard.Dashboard',
    xtype: 'p5-dashboard',
    

    initComponent: function() {
        var me = this;
    
        var columnCount = parseInt(document.dashboardForm.columnCount.value) || 1;
        var columnWidths = new Array(columnCount).fill(1/columnCount);
        var defaultContent = [];
        var widgetsValue = document.dashboardForm.widgets.value;
        var positionsValue = document.dashboardForm.widgetPositions.value;
        var isLicensed = !!+document.dashboardForm.isLicensed.value;
        var isConfigured = !!+document.dashboardForm.isConfigured.value;
        var state = Ext.state.Manager.get(me.stateId);
    
        // Widget-Liste bestimmen
        var widgets;
        // Initial setup oder wenn Lizenz fehlt
        if (!isLicensed) {
            Ext.state.Manager.clear(me.stateId);
            widgets = ['welcome-screen'];
        } 
        // Wenn Lizenz aber keine Konfiguration
        else if (!isConfigured) {
            Ext.state.Manager.clear(me.stateId);
            widgets = ['welcome-screen'];
        } 
        // Only change if welcome screen is the only widget
        // But only if isLicensed and isConfigured are true
        // see above
        else if (state && state.items && 
                state.items.length === 1 && 
                state.items[0].type === 'welcome-screen') {
            Ext.state.Manager.clear(me.stateId);
            widgets = ['installation-info', 'license-usage'];
        }
        // Normale Widget-Auswahl
        else if (widgetsValue && widgetsValue !== "[null]") {
            widgets = Ext.JSON.decode(widgetsValue);
        } 
        // Standard-Widgets als Fallback
        else {
            widgets = ['installation-info', 'license-usage'];
        }
    
        // Widget-Positionen organisieren
        var widgetsByColumn = new Array(columnCount);
        for (var i = 0; i < columnCount; i++) {
            widgetsByColumn[i] = [];
        }
    
        // Wenn wir gespeicherte Positionen haben, diese verwenden
        if (positionsValue && positionsValue !== "{}") {
            try {
                var widgetPositions = JSON.parse(positionsValue);
                
                // Widgets nach Spalten organisieren
                for (var widgetType in widgetPositions) {
                    if (widgets.indexOf(widgetType) !== -1) { // Nur aktive Widgets
                        var colIndex = widgetPositions[widgetType];
                        if (colIndex >= columnCount) colIndex = 0; // Sicherheitsprüfung
                        widgetsByColumn[colIndex].push(widgetType);
                    }
                }
            } catch(e) {
                console.error('Error parsing widget positions:', e);
            }
        } 
        // Fallback: Modulo-Verteilung, wenn keine Positionsdaten vorhanden sind
        else {
            widgets.forEach(function(widgetType, widgetIndex) {
                var columnIndex = widgetIndex % columnCount;
                widgetsByColumn[columnIndex].push(widgetType);
            });
        }
        
        // Jetzt alle Spalteninhalte in defaultContent umwandeln
        for (var colIndex = 0; colIndex < columnCount; colIndex++) {
            widgetsByColumn[colIndex].forEach(function(widgetType) {
                defaultContent.push({
                    type: widgetType,
                    columnIndex: colIndex,
                    id: 'dashboard-' + document.dashboardForm.module.value + '-' + widgetType
                });
            });
        }
    
        Ext.applyIf(me, {
            columnWidths: columnWidths,
            defaultContent: defaultContent
        });
    
        me.callParent();
    },
    stateEvents: ['drop', 'resize'],
    stateful: true,    
    stateId: 'dashboard-layout',
    padding: '0',
    border: false,
       
    parts: {
        'installation-info': {
            viewTemplate: {
                title: $$('P5 Installation'),
                header: { padding: '2 10 2 5' },
                closable: false,
                collapsible: false,
                items: [{
                    xtype: 'installation-info-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'license-usage': {
            viewTemplate: {
                title: 'P5 '+ $$('License usage'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }],
                items: [{
                    xtype: 'license-usage-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'notification': {
            viewTemplate: {
                title: 'P5 '+ $$('Notifications'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }],
                items: [{
                    xtype: 'notification-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'backup-stats': {
            viewTemplate: {
                title: $$('Backup Statistics'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'backup-stats-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'synchronize-stats': {
            viewTemplate: {
                title: $$('Synchronize Statistics'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'synchronize-stats-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'archive-stats': {
            viewTemplate: {
                title: $$('Archive Statistics'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'archive-stats-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'restore-stats': {
            viewTemplate: {
                title: $$('Restore Statistics'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'restore-stats-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'job-stats': {
            viewTemplate: {
                title: $$('Statistics on Job Results'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }],
                items: [{
                    xtype: 'job-stats-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'pool-usage': {
            viewTemplate: {
                title: $$('Tape Pool Usage'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'pool-usage-widget',
                    border: 0,
                    height: 197
                }]
            }
        },            
        'clouddisk-usage': {
            viewTemplate: {
                title: $$('Cloud/Disk Pool Usage'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'clouddisk-usage-widget',
                    border: 0,
                    height: 197
                }]
            }
        },      
        'drive-speed': {
            viewTemplate: {
                title: $$('Tape Write Performance'),
                header: { padding: '2 10 2 5' },
                closable: false,
                items: [{
                    xtype: 'drive-speed-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'welcome-screen': {
            viewTemplate: {
                //header: { padding: '2 10 2 5' },
                closable: false,
                collapsible: false,
                items: [{
                    xtype: 'welcome-screen-widget',
                    border: 0,
                    height: 426
                }]
            }
        },
        'backup-overview': {
            viewTemplate: {
                title: $$('Backup Overview'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'backup-overview-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'synchronize-overview': {
            viewTemplate: {
                title: $$('Synchronize Overview'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'synchronize-overview-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'archive-overview': {
            viewTemplate: {
                title: $$('Archive Overview'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'archive-overview-widget',
                    border: 0,
                    height: 197
                }]
            }
        },
        'index-usage': {
            viewTemplate: {
                title: $$('Index Storage Analysis'),
                header: { padding: '2 10 2 5' },
                closable: false,
                tools: [{
                    type: 'refresh',
                    handler: 'onWidgetRefresh'
                }, {
                    type: 'save',
                    tooltip: $$('Export as TSV'),
                    handler: 'onWidgetExport'
                }],
                items: [{
                    xtype: 'archive-index-usage-widget',
                    border: 0,
                    height: 426
                }]
            }
        }
    },
    tbar: [{
        xtype: 'component',
        html: '<div class="t30">Archiware P5 Dashboard</div>',
        flex: 1,  // Takes up available space on the left
        listeners: {
            element: 'el',
            click: 'activateDemoMode'
        }
    },{
        text: $$('Columns'),
        bind: {
            disabled: '{!(isLicensed && isConfigured)}'
        },
        menu: {
            plain: true,
            items: [{
                xtype: 'radiogroup',
                columns: 1,
                vertical: true,
                listeners: {
                    beforerender: function(group) {
                        var dashboard = group.up('dashboard');
                        var state = Ext.state.Manager.get(dashboard.stateId);
                        var columnCount = state ? state.columnWidths.length : 1;
                        group.setValue({ columns: columnCount });
                    },
                    change: 'onColumnRadioChange'
                },
                items: [{
                    boxLabel: '1 '+$$('Column'),
                    name: 'columns',
                    inputValue: 1,
                }, {
                    boxLabel: '2 '+$$('Columns'),
                    name: 'columns',
                    inputValue: 2,
                }, {
                    boxLabel: '3 '+$$('Columns'),
                    name: 'columns',
                    inputValue: 3,
                }, {
                    boxLabel: '4 '+$$('Columns'),
                    name: 'columns',
                    inputValue: 4,
                }]
            }]
        }
    },{
        text: $$('Widgets'),
        bind: {
            disabled: '{!(isLicensed && isConfigured)}'
        },
        menu: {
            items: [{
                xtype: 'checkboxgroup',
                columns: 1,
                vertical: true,
                listeners: {
                    beforerender: function(group) {
                        var dashboard = group.up('dashboard');
                        var state = Ext.state.Manager.get(dashboard.stateId);
                        var widgets = state ? state.items.map(function(w) { return w.type; }) : [];
                        group.setValue({
                            widgets: widgets
                        });
                    },
                    change: 'onWidgetCheckChange'
                },
                items: [{
                    xtype: 'component',
                    cls: 'menu-group-text',
                    html: 'Installation',
                }, {
                    boxLabel: $$('P5 Installation'),
                    name: 'widgets',
                    inputValue: 'installation-info',
                    disabled: true // These widgets are always active
                }, {
                    boxLabel: $$('P5 License Usage'),
                    name: 'widgets',
                    inputValue: 'license-usage',
                }, {
                    boxLabel: $$('Notifications'),
                    name: 'widgets',
                    reference: 'notificationckbox_ref',
                    inputValue: 'notification',
                }, {
                    boxLabel: $$('Welcome Widget'),
                    name: 'widgets',
                    inputValue: 'welcome-screen'
                }, {
                    xtype: 'component',
                    cls: 'menu-group-text',
                    html: 'Jobs'
                }, {
                    boxLabel: $$('Backup Statistics'),
                    name: 'widgets',
                    inputValue: 'backup-stats'
                }, {
                    boxLabel: $$('Synchronize Statistics'),
                    name: 'widgets',
                    inputValue: 'synchronize-stats'
                }, {
                    boxLabel: $$('Archive Statistics'),
                    name: 'widgets',
                    inputValue: 'archive-stats'
                }, {
                    boxLabel: $$('Restore Statistics'),
                    name: 'widgets',
                    inputValue: 'restore-stats'
                }, {
                    boxLabel: $$('Job Results'),
                    name: 'widgets',
                    inputValue: 'job-stats'
                }, {
                    xtype: 'component',
                    cls: 'menu-group-text',
                    html: 'Storage',
                }, {
                    boxLabel: $$('Tape Pool Usage'),
                    name: 'widgets',
                    inputValue: 'pool-usage'
                }, {
                    boxLabel: $$('Cloud/Disk Pool Usage'),
                    name: 'widgets',
                    inputValue: 'clouddisk-usage'
                }, {
                    boxLabel: $$('Tape Performance'),
                    name: 'widgets',
                    inputValue: 'drive-speed'
                }, {
                    xtype: 'component',
                    cls: 'menu-group-text',
                    html: 'Job Details',
                }, {
                    boxLabel: $$('Backup Overview'),
                    name: 'widgets',
                    inputValue: 'backup-overview'
                }, {
                    boxLabel: $$('Synchronize Overview'),
                    name: 'widgets',
                    inputValue: 'synchronize-overview'
                }, {
                    boxLabel: $$('Archive Overview'),
                    name: 'widgets',
                    inputValue: 'archive-overview'
                }, {
                    xtype: 'component',
                    cls: 'menu-group-text',
                    html: 'DLM Utilization',
                }, {
                    boxLabel: $$('Index Storage Analysis'),
                    name: 'widgets',
                    inputValue: 'index-usage'
                }]
            }, '-', {
                text: $$('Reset Dashboard'),
                handler: 'onDashboardReset'
            }]
         }
    },{
        xtype: 'component',
        itemId: 'notificationIcon',
        cls: 'fas fa-bell',
        style: {
            cursor: 'pointer'
        },
        bind: {
            disabled: '{!(isLicensed && isConfigured)}',
            style: {
                fontSize: '14px',
                color: '{hasNotifications ? "#FF0000" : "#666666"}',
                marginRight: '10px'
            }
        },
        listeners: {
            element: 'el',
            click: 'onNotificationClick'
        }
    }]
});