Ext.define('P5Index.view.PathImportWindowController', {
    extend: 'Ext.app.ViewController',
    alias: 'controller.pathimportwindow',
    
    // Konfiguration für Batch Processing
    MAX_CONCURRENT_REQUESTS: 5,

    // Batch-Processing Variablen
    activeBatchRequests: 0,
    processedItems: 0,
    isBatchProcessing: false,
    
    debugMode: false,  // NEU: Als Property für konsistentes Debug-Logging
    
    setupWindowDropZone: function() {
        var me = this;
        var el = this.getView().getEl();
        
        el.on('dragover', function(e) {
            e.preventDefault();
        });
        
        el.on('drop', function(e) {
            e.preventDefault();
            var files = e.browserEvent.dataTransfer.files;
            if (files.length > 0) {
                me.processPathFile(files[0]);
            }
        });
    },
    
    // Path Datei Dialog öffnen
    openPathFileDialog: function() {
        var me = this;
        var fileInput = document.createElement('input');
        fileInput.type = 'file';
        fileInput.accept = '.otio,.json,.xml,.fcpxml,.edl,.ale,.txt,.tsv,.log,.aaf';
        
        fileInput.onchange = function(e) {
            if (e.target.files.length > 0) {
                me.processPathFile(e.target.files[0]);
            }
        };
        
        fileInput.click();
    },
    
    // Path Datei verarbeiten
    processPathFile: function(file) {
        var me = this;
        
        var logger = P5Index.util.ParserLogger;
        
        //check if logging is enabled
        var enableLogging = this.lookupReference('enableLogging').getValue();
        if (enableLogging) {
            logger.startLogging();
            // Flag für Server-Requests setzen
            this.loggingEnabled = true;
            
            // NEU: Debug-Mode check
            var debugCheckbox = this.lookupReference('enableLogging');
            if (debugCheckbox.boxLabel && debugCheckbox.boxLabel.includes('Debug')) {
                this.debugMode = true;
            }
        }

        // UI für Analyse vorbereiten
        me.showAnalysisUI();
        
        // Define statusText
        var statusText = this.lookupReference('statusText');
        
        // File-Size Check
        var sizeInMB = file.size / (1024 * 1024);
        if (sizeInMB > 100) {
            // NEU: Log oversized file
            if (logger.isLogging) {
                logger.addLog('File rejected: ' + file.name + ' (' + sizeInMB.toFixed(2) + 'MB) - exceeds 100MB limit', 'ERROR');
            }
            Ext.Msg.alert('Error', 'File too large (max 100MB). Please split into smaller files.');
            return;
        }
        
        // Dateityp ermitteln
        var fileExt = file.name.split('.').pop().toLowerCase();
        var startTime = Date.now();  // NEU: Zeit messen
        
        // NEU: Log file info
        if (this.loggingEnabled) {
            logger.addLog('=== PARSE: ' + file.name + ' (' + me.formatFileSize(file.size) + ') ===');
        }

        // Package-Formate früh abfangen
        if (fileExt === 'fcpxmld') {
            // NEU: Log package format
            if (logger.isLogging) {
                logger.addLog('Package format detected: ' + file.name + ' - user guidance shown', 'WARNING');
            }
            Ext.Msg.alert('Package Format', 
                'FCPXMLD packages cannot be read directly.<br/><br/>' +
                'Please:<br/>' +
                '1. Right-click the .fcpxmld file<br/>' +
                '2. Select "Show Package Contents"<br/>' +
                '3. Drag the Info.fcpxml file instead'
            );
            return;
        }

        var supportedFormats = ['otio', 'json', 'xml', 'fcpxml','edl', 'ale', 'txt', 'tsv', 'log', 'aaf'];
        
        if (!supportedFormats.includes(fileExt)) {
            // NEU: Log unsupported format
            if (logger.isLogging) {
                logger.addLog('Unsupported format: .' + fileExt, 'ERROR');
            }
            statusText.setText('<i class="fa fa-exclamation-triangle" style="color: #ff8c00; margin-right: 5px;"></i>Unsupported file format');
            Ext.Msg.alert('Error', 'Please upload a supported project file: ' + supportedFormats.join(', '));
            return;
        }
            
            // Ab hier war statusText bereits definiert
        statusText.setText('Analyzing project file: ' + file.name);
        
        // Datei mit FileReader lesen
        var reader = new FileReader();
        
        reader.onload = function(e) {
            try {
                var fileContent = e.target.result;
                var result = me.parseFileContent(fileContent, fileExt, file.name);
                
                if (me.loggingEnabled) {
                    var duration = Date.now() - startTime;
                    
                    if (result.success) {
                        logger.addLog('Parse completed: ' + result.totalCount + ' items in ' + duration + 'ms');
                        
                        // Filter info wenn items entfernt wurden
                        if (result.filteredItems > 0) {
                            logger.addLog('Filtered: ' + result.filteredItems + ' items (transitions/effects)');
                        }
                        
                        // Sample items
                        if (result.mediaItems.length > 0) {
                            var samples = result.mediaItems.slice(0, 3).map(function(item) {
                                return item.fileName;
                            }).join(', ');
                            if (result.mediaItems.length > 3) {
                                samples += '...';
                            }
                            logger.addLog('Sample: ' + samples);
                        }
                    } else {
                        logger.addLog('Parse failed - ' + result.error, 'ERROR');
                    }
                }

                if (!result.success) {
                    throw new Error(result.error || 'Unknown error parsing file');
                }

                // Window-Titel mit Filename erweitern
                me.getView().setTitle('Media Recovery - ' + file.name);

                // Media-Items für spätere Archive-Suche speichern
                me.parsedMediaItems = result.mediaItems;
                console.log('Parsed media items:', me.parsedMediaItems.length, 'items');

                // Intelligente Filter-Defaults basierend auf Parse-Ergebnis
                me.setSmartFilterDefaults(result);
                
                me.loadProjectDataToStore();
                
                // Status aktualisieren
                var statusMessage = 'File analyzed: ' + file.name + ' (' + result.totalCount + ' media references found)';
                if (result.detectedFormat) {
                    statusMessage += ' - Format: ' + result.detectedFormat;
                }
                statusText.setText(statusMessage);
                
                // Such-Button aktivieren
                me.lookupReference('searchBtn').setDisabled(false);
                me.lookupReference('resetBtn').setDisabled(false);
                
                // Automatische Suche starten
                me.searchArchiveForPathItems();
                
            } catch (error) {
                console.error('Error parsing project file:', error);

                if (me.loggingEnabled) {
                    logger.addLog('ERROR: ' + error.message, 'ERROR');
                }

                statusText.setText('<i class="fa fa-exclamation-circle" style="color: #dc3545; margin-right: 5px;"></i>Error: ' + error.message);
                Ext.Msg.alert('Error', 'File could not be parsed: ' + error.message);
            }
        };
        
        reader.onerror = function() {
            if (me.loggingEnabled) {
                logger.addLog('ERROR: File read failed', 'ERROR');
            }

            var statusText = me.lookupReference('statusText');
            statusText.setText('<i class="fa fa-exclamation-circle" style="color: #dc3545; margin-right: 5px;"></i>Error reading file');
            Ext.Msg.alert('Error', 'File could not be read');
        };
        
        reader.readAsText(file);
    },
    
    // Dateiinhalt basierend auf Format parsen
    parseFileContent: function(content, fileExt, fileName) {
        var vm = this.getViewModel();
        
        try {
            console.log(`🔍 Parsing file: ${fileName} (${fileExt.toUpperCase()})`);
            
            // NEUE OPTIMIERTE PARSER MIT CUSTOMER FILTER
            var result = P5Index.util.OptimizedMediaParser.parseForArchiveIndex(content, fileExt, fileName);
            
            if (result.success) {
                var formatName = this.extractFormatName(result.detectedFormat);
                var hasTimecodes = this.formatSupportsTimecodes(formatName);
                var hasAvailableDuration = this.formatSupportsAvailableDuration(formatName);
                
                vm.set('showUsedDurationTC', hasTimecodes);
                vm.set('showAvailableDurationTC', hasAvailableDuration);
                
                if (result.filteredItems > 0) {
                    console.log(`Customer filter: Removed ${result.filteredItems} non-media items from ${fileName}`);
                }
            }
            
            return result;
            
        } catch (error) {
            console.error(`Fatal error parsing ${fileName}:`, error);
            return {
                success: false,
                error: `Unexpected error: ${error.message}`
            };
        }
    },

    // NEUE HILFSFUNKTIONEN HINZUFÜGEN:
    extractFormatName: function(detectedFormat) {
        if (!detectedFormat) return '';
        var format = detectedFormat.split(' ')[0];
        if (detectedFormat.includes('FCPXML')) return 'FCPXML';
        if (detectedFormat.includes('XMEML')) return 'XMEML';
        if (detectedFormat.includes('OTIO')) return 'OTIO';
        if (detectedFormat.includes('Avid')) return 'ALE';
        return format;
    },

    formatSupportsTimecodes: function(formatName) {
        var timecodeFormats = ['CMX', 'EDL', 'ALE', 'Avid', 'FCPXML', 'XMEML', 'OTIO'];
        return timecodeFormats.includes(formatName);
    },

    formatSupportsAvailableDuration: function(formatName) {
        var availableDurationFormats = ['FCPXML', 'XMEML', 'OTIO'];
        return availableDurationFormats.includes(formatName);
    },
    
    // ===== ROWWIDGET ANALYSIS UI METHODS =====
    
    // Analysis UI anzeigen
    showAnalysisUI: function() {
        this.lookupReference('statusToolbar').setVisible(true);
        this.lookupReference('pathMediaGrid').setVisible(true);
    },
    
    // Daten in Projekt-Store laden
    loadProjectDataToStore: function() {
        var me = this;
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var logger = P5Index.util.ParserLogger;  // NEU
        
        if (this.parsedMediaItems && this.parsedMediaItems.length > 0) {
            projectStore.removeAll();
            
            // NEU: Log store loading
            if (logger.isLogging && this.debugMode) {
                logger.addLog('Loading ' + this.parsedMediaItems.length + ' items to project store');
            }
            
            this.parsedMediaItems.forEach(function(mediaItem) {
                // Archive-Versions-Store für diese Datei erstellen
                var archiveStore = me.getViewModel().createArchiveVersionsStore();
                
                var projectRecord = {
                    fileName: mediaItem.fileName,
                    usedDurationTC: mediaItem.usedDurationTC || '',
                    availableDurationTC: mediaItem.availableDurationTC || '',
                    sourceFormat: mediaItem.sourceFormat,
                    originalPath: mediaItem.path,
                    foundVersionCount: 0,
                    archiveVersionsStore: archiveStore
                };
                
                projectStore.add(projectRecord);
            });
        }
    },
    
    // ===== ARCHIVE SEARCH METHODS (BATCH PROCESSING) =====
    
    // Archiv nach Path-Items durchsuchen
    searchArchiveForPathItems: function() {
        var me = this;
        var statusText = me.lookupReference('statusText');
        var logger = P5Index.util.ParserLogger;  // NEU
        
        me.getView().setLoading('Searching archive...');
        statusText.setText('Searching archive...');

        if (me.loggingEnabled) {
            logger.addLog('=== SEARCH STARTED ===', 'SEARCH');
            
            // NEU: Log active filters
            var filters = this.getActiveFilters();
            logger.addLog('Search filters: pathMatching=' + filters.pathMatching + 
                         ', searchMode=' + filters.searchMode);
        }
        
        if (!me.parsedMediaItems || me.parsedMediaItems.length === 0) {
            // EDGE CASE: 0 items - sofort beenden
            // NEU: Log edge case
            if (logger.isLogging) {
                logger.addLog('No items to search - aborting', 'WARNING');
            }
            me.getView().setLoading(false);
            statusText.setText('<i class="fa fa-exclamation-triangle" style="color: #ff8c00; margin-right: 5px;"></i>No media references found to search for');
            me.updateButtonStates();
            return;
        }
        
        // Normale Batch-Verarbeitung für > 0 items
        me.searchCount = 0;
        me.totalSearches = me.parsedMediaItems.length;
        me.activeBatchRequests = 0;
        me.processedItems = 0;
        me.isBatchProcessing = true;
        
        // NEU: Log batch start
        if (logger.isLogging) {
            logger.addLog('Starting batch search for ' + me.totalSearches + ' items (max ' + 
                         me.MAX_CONCURRENT_REQUESTS + ' concurrent)');
        }
        
        console.log('Starting batch processing for', me.totalSearches, 'items');
        me.processBatch();
    },

    // Batch-Verarbeitung mit begrenzter Parallelität
    processBatch: function() {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        
        if (!this.isBatchProcessing) {
            return; // Abgebrochen
        }
        
        // Berechne wie viele neue Requests gestartet werden können
        var availableSlots = this.MAX_CONCURRENT_REQUESTS - this.activeBatchRequests;
        var remainingItems = this.totalSearches - this.processedItems;
        var itemsToProcess = Math.min(availableSlots, remainingItems);
        
        // NEU: Debug log batch details
        if (logger.isLogging && this.debugMode) {
            logger.addLog('Batch: active=' + this.activeBatchRequests + 
                         ', slots=' + availableSlots + 
                         ', toProcess=' + itemsToProcess);
        }
        
        console.log('Processing batch: active=', this.activeBatchRequests, 
                    'available=', availableSlots, 'toProcess=', itemsToProcess);
        
        // Neue Requests starten
        for (var i = 0; i < itemsToProcess; i++) {
            var itemIndex = this.processedItems + i;
            if (itemIndex < this.totalSearches) {
                var mediaItem = this.parsedMediaItems[itemIndex];
                
                // Request-Counter erhöhen
                this.activeBatchRequests++;
                
                // Request starten
                this.searchInArchiveIndexBatch(mediaItem, itemIndex);
            }
        }
        
        // Processed counter aktualisieren
        this.processedItems += itemsToProcess;
        
        // Progress nach Request-Start aktualisieren
        if (itemsToProcess > 0) {
            this.updateBatchProgress();
        }
    },

    // Filter-Parameter sammeln
    getActiveFilters: function() {
        return {
            pathMatching: this.lookupReference('pathMatchingCombo').getValue(),
            searchMode: this.lookupReference('searchModeCombo').getValue(),
            // filesOnly: this.lookupReference('filesOnlyCheck').getValue(),
        };
    },

    // Einzelnes Media-Item im Archiv-Index suchen (Batch-Version)
    searchInArchiveIndexBatch: function(mediaItem, itemIndex) {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var projectRecord = projectStore.getAt(itemIndex);
        
        if (!projectRecord) {
            console.error('No project record found at index:', itemIndex);
            me.handleBatchItemComplete(itemIndex, false);
            return;
        }
        
        // Store-interne ID als Request-ID verwenden
        var requestId = projectRecord.getId();
        var filters = this.getActiveFilters();
        
        // NEU: Debug log individual search
        if (logger.isLogging && this.debugMode) {
            logger.addLog('Search [' + itemIndex + ']: "' + mediaItem.fileName + '"');
        }
        
        Ext.Ajax.request({
            url: browser_index,
            method: 'POST',
            timeout: 60000, // 60 Sekunden pro Request
            params: {
                caller: 'ajaxVerifyPath',
                fileName: mediaItem.fileName,
                path: mediaItem.path,
                requestId: requestId,
                pathMatching: filters.pathMatching,
                searchMode: filters.searchMode,
                enableDebugLog: me.loggingEnabled ? 1 : 0
            },
            success: function(response) {
                try {
                    var result = Ext.decode(response.responseText);
                    var returnedRequestId = result.requestId || requestId;
                    
                    // NEU: Server-Logs in separaten Store speichern
                    if (me.loggingEnabled && result.debugLogs) {
                        projectRecord.set('searchQuery', result.finalQuery || '');
                        projectRecord.set('searchMode', result.searchStrategy || '');
                        projectRecord.set('pathMatching', filters.pathMatching);
                        projectRecord.set('serverLogs', result.debugLogs);
                        projectRecord.set('timestamp', new Date());
                        projectRecord.commit();
                    }

                    if (result.success && result.versions && result.versions.length > 0) {
                        // NEU: Log found items
                        if (logger.isLogging && me.debugMode) {
                            logger.addLog('Found ' + result.versions.length + ' versions for "' + 
                                        mediaItem.fileName + '"');
                        }
                        me.addArchiveVersionsToStoreByRecordId(returnedRequestId, result.versions);
                    } else {
                        if (projectRecord) {
                            projectRecord.set('foundVersionCount', 0);
                        }
                    }
                } catch (e) {
                    console.error('Error parsing response:', e);
                    // NEU: Log parse error
                    if (logger.isLogging) {
                        logger.addLog('Response parse error for "' + mediaItem.fileName + '": ' + 
                                    e.message, 'ERROR');
                    }
                    if (projectRecord) {
                        projectRecord.set('foundVersionCount', 0);
                    }
                }
                // Process batch completion
                me.handleBatchItemComplete(itemIndex, true);
            },
            failure: function(response) {
                console.error('AJAX request failed/timeout for item:', itemIndex, response);
                // NEU: Log request failure
                if (logger.isLogging) {
                    logger.addLog('Search failed for "' + mediaItem.fileName + '": ' + 
                                (response.statusText || 'timeout'), 'ERROR');
                }
                //var projectRecord = projectStore.getById(requestId);
                if (projectRecord) {
                    projectRecord.set('foundVersionCount', 0);
                }
                me.handleBatchItemComplete(itemIndex, false);
            }
        });
    },

    // Process individual batch item completion
    handleBatchItemComplete: function(itemIndex, success) {
        // Reduce request counters
        this.activeBatchRequests--;
        this.searchCount++;
        
        console.log('Item', itemIndex, 'completed. Active requests:', this.activeBatchRequests, 
                    'Progress:', this.searchCount, '/', this.totalSearches);
        
        // Update progress
        this.updateBatchProgress();
        
        // Check whether further items can be processed
        if (this.isBatchProcessing && this.processedItems < this.totalSearches) {
            // More items in the queue - start next batch
            this.processBatch();
        } else if (this.activeBatchRequests === 0) {
            // All requests completed
            this.completeBatchProcessing();
        }
    },

    // Update batch progress
    updateBatchProgress: function() {
        var statusText = this.lookupReference('statusText');
        var percentage = Math.round((this.searchCount / this.totalSearches) * 100);
        
        statusText.setText(`Searching archive... ${this.searchCount}/${this.totalSearches} (${percentage}%) - Active: ${this.activeBatchRequests}`);
    },

    // Complete batch processing
    completeBatchProcessing: function() {
        var logger = P5Index.util.ParserLogger;
        
        console.log('Batch processing completed');
        
        this.isBatchProcessing = false;
        this.getView().setLoading(false);
        
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var totalProjects = projectStore.getCount();
        var foundProjects = 0;
        var totalVersions = 0;
        var directoryCount = 0;  // NEU
        
        projectStore.each(function(record) {
            var versionCount = record.get('foundVersionCount');
            if (versionCount > 0) {
                foundProjects++;
                totalVersions += versionCount;
                
                // NEU: Zähle Directories
                var archiveStore = record.get('archiveVersionsStore');
                if (archiveStore) {
                    archiveStore.each(function(version) {
                        if (version.get('isDirectory')) {
                            directoryCount++;
                        }
                    });
                }
            }
        });
        
        // Log search summary
        if (logger.isLogging) {
            logger.addLog('=== SEARCH COMPLETED ===', 'SEARCH');
            logger.addLog('Found ' + foundProjects + ' of ' + totalProjects + ' files in archive');
            logger.addLog('Total versions found: ' + totalVersions);
            if (directoryCount > 0) {
                logger.addLog('Including ' + directoryCount + ' directories');
            }
            var successRate = totalProjects > 0 ? Math.round((foundProjects/totalProjects)*100) : 0;
            logger.addLog('Success rate: ' + successRate + '%');
        }
        
        var statusText = this.lookupReference('statusText');
        var statusMessage = 'Search completed: ' + foundProjects + ' of ' + totalProjects + ' files found in archive';
        
        // NEU: Erwähne Directories wenn vorhanden
        if (directoryCount > 0) {
            statusMessage += ' (including ' + directoryCount + ' director' + (directoryCount === 1 ? 'y' : 'ies') + ')';
        }
        
        statusText.setText(statusMessage);

        // Switch UI if logging was active
        if (this.loggingEnabled) {
            this.lookupReference('enableLogging').hide();
            this.lookupReference('showLogBtn').show();
            P5Index.util.ParserLogger.stopLogging();
        }
        
        // Auto-select latest versions
        this.autoSelectLatestVersions();
        this.updateButtonStates();
    },

    // Cancel batch processing (for reset)
    cancelBatchProcessing: function() {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        // NEU: Log cancellation
        if (logger.isLogging && this.isBatchProcessing) {
            logger.addLog('Batch processing cancelled', 'WARNING');
        }
        
        this.isBatchProcessing = false;
        this.activeBatchRequests = 0;
        console.log('Batch processing cancelled');
    },
    
    // Add archive versions to RowWidget store (record ID based)
    addArchiveVersionsToStoreByRecordId: function(recordId, archiveVersions) {
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        
        // Direct access via store-internal ID
        var projectRecord = projectStore.getById(recordId);
        
        if (projectRecord) {
            var archiveStore = projectRecord.get('archiveVersionsStore');
            // Add archive versions
            archiveVersions.forEach(function(version) {
                var storeRecord = {
                    fileName: projectRecord.get('fileName'),
                    foundFileName: version.foundName,
                    selected: false,
                    atime: version.atime,
                    size: version.size || 0,
                    relativePath: version.relativePath || '',
                    archiveLocation: version.archiveLocation || '',
                    elementUrl: version.elementUrl || '',
                    ranking: version.ranking || 0,
                    isDirectory: version.isDirectory || 0
                };
                
                archiveStore.add(storeRecord);
            });
            
            // Auto-select neueste Version
            this.autoSelectLatestVersionForProject(projectRecord);
            
            // Statistiken aktualisieren
            projectRecord.set('foundVersionCount', archiveVersions.length);
            
            console.log('Successfully mapped recordId', recordId, 'to project record');
        } else {
            console.warn('No project record found for recordId:', recordId);
        }
    },
    
    // ===== SELECTION AND UI METHODS (ADAPTED FOR ROWWIDGET) =====
    
    // Auto-select latest version for each file group
    autoSelectLatestVersions: function() {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var selectedCount = 0;  // NEU
        
        projectStore.each(function(projectRecord) {
            if (me.autoSelectLatestVersionForProject(projectRecord)) {
                selectedCount++;  // NEU
            }
        });
        
        // NEU: Log auto-selection
        if (logger.isLogging && this.debugMode) {
            logger.addLog('Auto-selected ' + selectedCount + ' latest versions');
        }
    },
    
    // Auto-select für einzelnes Projekt
    autoSelectLatestVersionForProject: function(projectRecord) {
        var archiveStore = projectRecord.get('archiveVersionsStore');
        
        if (archiveStore.getCount() > 0) {
            // Finde Eintrag mit höchstem Ranking (neueste Version)
            var bestRecord = null;
            var bestRanking = 0;
            
            archiveStore.each(function(record) {
                var ranking = record.get('ranking');
                var isDirectory = record.get('isDirectory');
                
                // NEU: Skip directories bei Auto-Selection
                if (isDirectory) {
                    return;
                }
                
                if (ranking > bestRanking) {
                    bestRanking = ranking;
                    bestRecord = record;
                }
            });
            
            // Wähle den besten FILE-Eintrag aus (keine Directories)
            if (bestRecord && bestRanking > 0) {
                bestRecord.set('selected', true);
                return true;
            }
        }
        return false;
    },
    
    // Handle checkbox change in archive versions
    onArchiveVersionCheckChange: function(column, rowIndex, checked, record) {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        if (checked) {
            // Alle anderen in diesem Store deselektieren
            var store = record.store;
            store.each(function(otherRecord) {
                if (otherRecord !== record) {
                    otherRecord.set('selected', false);
                }
            });
        }
        
        record.set('selected', checked);
        
        // NEU: Debug log selection change
        if (logger.isLogging && this.debugMode && checked) {
            logger.addLog('Manual selection: "' + record.get('foundFileName') + 
                         '" for "' + record.get('fileName') + '"');
        }
        
        this.updateButtonStates();
    },

    // Button-States basierend auf allen ausgewählten Archive-Versionen
    updateButtonStates: function() {
        var me = this;
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var hasSelected = false;
        var selectedCount = 0;
        
        // Zähle ALLE selected items
        projectStore.each(function(projectRecord) {
            var archiveStore = projectRecord.get('archiveVersionsStore');
            if (archiveStore) {
                archiveStore.each(function(archiveRecord) {
                    if (archiveRecord.get('selected') && archiveRecord.get('ranking') > 0) {
                        hasSelected = true;
                        selectedCount++;
                    }
                });
            }
        });
        
        this.lookupReference('addSelectedBtn').setDisabled(!hasSelected);
        
        // Update button label
        if (selectedCount === 0) {
            this.lookupReference('addSelectedBtn').setText('Restore');
        } else if (selectedCount === 1) {
            this.lookupReference('addSelectedBtn').setText('Restore 1 File');
        } else {
            this.lookupReference('addSelectedBtn').setText('Restore ' + selectedCount + ' Files');
        }
    },
    
    // Intelligente Filter-Defaults
    setSmartFilterDefaults: function(parseResult) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var hasDetailedPaths = false;
        var hasExtensions = false;
        var avgDepth = 0;
        var pathCount = 0;
        
        if (parseResult.mediaItems) {
            parseResult.mediaItems.forEach(function(item) {
                // Pfad-Analyse
                if (item.path) {
                    var pathSegments = item.path.split('/').length;
                    avgDepth += pathSegments;
                    pathCount++;
                    
                    // Detaillierte Pfade erkennen
                    if (pathSegments > 3) {
                        hasDetailedPaths = true;
                    }
                }
                
                // Extension-Analyse
                if (item.fileName && item.fileName.includes('.')) {
                    hasExtensions = true;
                }
            });
            
            if (pathCount > 0) {
                avgDepth = avgDepth / pathCount;
            }
        }
        
        // Path Matching Default
        var defaultPathMatching = 'parent';  // Safe default
        if (avgDepth > 4 && hasDetailedPaths) {
            defaultPathMatching = 'relative';
        } else if (avgDepth <= 2) {
            defaultPathMatching = 'ignore';
        }
        
        // Filter-Defaults setzen
        this.lookupReference('pathMatchingCombo').setValue(defaultPathMatching);
        this.lookupReference('searchModeCombo').setValue('exact');
        
        // NEU: Log smart defaults
        if (logger.isLogging && this.debugMode) {
            logger.addLog('Smart filter defaults: pathMatching=' + defaultPathMatching + 
                         ', avgPathDepth=' + avgDepth.toFixed(1) + 
                         ', hasExtensions=' + hasExtensions);
        }
        
        console.log('Smart defaults:', {
            pathMatching: defaultPathMatching,
            extension: hasExtensions,
            avgDepth: avgDepth.toFixed(1)
        });
    },
    
    // Event-Handler für Projekt-Selektion
    onProjectSelectionChange: function(selModel, selected) {
        // Optional: Zusätzliche Logik für Projekt-Auswahl
        console.log('Project selection changed:', selected.length);
    },
    
    onSearchArchive: function() {
        var me = this;
        var logger = P5Index.util.ParserLogger;  // NEU
        
        if (!this.parsedMediaItems || this.parsedMediaItems.length === 0) {
            Ext.Msg.alert('Information', 'Please load a file first.');
            return;
        }
        
        // NEU: Log manual search
        if (logger.isLogging) {
            logger.addLog('Manual search initiated');
        }
        
        // Bestehende Archive-Stores leeren bei manueller Suche
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        projectStore.each(function(record) {
            var archiveStore = record.get('archiveVersionsStore');
            if (archiveStore) {
                archiveStore.removeAll();  // Vorherige Suchergebnisse löschen
            }
            record.set('foundVersionCount', 0);
        });
        
        // Button States zurücksetzen
        me.lookupReference('addSelectedBtn').setDisabled(true);
        
        // Neue Suche starten
        this.searchArchiveForPathItems();
    },
    
    onResetImport: function() {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        // NEU: Log reset
        if (logger.isLogging) {
            logger.addLog('=== RESET ===');
        }
        
        // Batch-Processing stoppen
        this.cancelBatchProcessing();
        
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        projectStore.removeAll();

        this.getView().setTitle('Media Recovery');
        this.getView().setLoading(false);

        this.lookupReference('statusText').setText('Ready to analyze');  // oder welcher Default-Text gewünscht ist

        this.lookupReference('searchBtn').setDisabled(true);
        this.lookupReference('resetBtn').setDisabled(true);
        this.lookupReference('addSelectedBtn').setDisabled(true);
        this.lookupReference('addSelectedBtn').setText('Restore');  // NEU: Reset label

        this.lookupReference('enableLogging').show();
        this.lookupReference('showLogBtn').hide();
        this.lookupReference('enableLogging').setValue(false);

        // Filter-Defaults setzen
        this.lookupReference('pathMatchingCombo').setValue('parent');
        this.lookupReference('searchModeCombo').setValue('exact');
        
        // Parser Logs leeren
        P5Index.util.ParserLogger.clearLogs();
        this.loggingEnabled = false;

        // Reset internal state
        this.parsedMediaItems = null;
        this.searchCount = 0;
        this.totalSearches = 0;
        this.activeBatchRequests = 0;
        this.processedItems = 0;
        this.isBatchProcessing = false;
        this.debugMode = false;  // NEU: Reset debug mode
    },
    
    // ===== ADD TO BASKET METHODS (ADAPTED FOR ROWWIDGET) =====
    
    // Ausgewählte Items sammeln
    onAddSelectedItems: function() {
        var logger = P5Index.util.ParserLogger;  // NEU
        var projectStore = this.getViewModel().getStore('pathProjectStore');
        var selectedItems = [];
        var selectedFileNames = [];  // NEU: für Log
        
        projectStore.each(function(projectRecord) {
            var archiveStore = projectRecord.get('archiveVersionsStore');
            if (archiveStore) {
                archiveStore.each(function(archiveRecord) {
                    if (archiveRecord.get('selected') && archiveRecord.get('ranking') > 0) {
                        selectedItems.push(archiveRecord.get('elementUrl'));
                        selectedFileNames.push(archiveRecord.get('foundFileName'));  // NEU
                    }
                });
            }
        });
        
        // NEU: Log restore action
        if (logger.isLogging) {
            logger.addLog('=== RESTORE ' + selectedItems.length + ' ITEMS ===');
            if (this.debugMode && selectedFileNames.length > 0) {
                var sample = selectedFileNames.slice(0, 5).join(', ');
                if (selectedFileNames.length > 5) {
                    sample += '... (' + (selectedFileNames.length - 5) + ' more)';
                }
                logger.addLog('Files: ' + sample);
            }
        }
        
        this.startRestore(selectedItems);
    },
    
    startRestore: function(items) {
        if (items.length === 0) {
            Ext.Msg.alert('Information', 'No items available to restore.');
            return;
        }

        var elementType = 'AsxTree';
        var aktion = 'dialog_restore';
        var url = document.urls.dialog_restore.value;

        win = window.open("", client[elementType][aktion].target, client[elementType][aktion].wsize);
        win.focus();
        winOpenClose(win);
        document.windowForm.action = url;
        document.windowForm.elementCall.value  = aktion;
        document.windowForm.target = client[elementType][aktion].target;
        document.windowForm.elementurl.value = Ext.encode(items);
        document.windowForm.submit();
    },

    // PathImportWindowController.js
    highlightDifferences: function(original, found) {
        // Einfacher Extension-Diff
        var origBase = this.getBaseName(original);
        var origExt = this.getExtension(original);
        var foundBase = this.getBaseName(found);
        var foundExt = this.getExtension(found);
        
        var result = '';
        
        // Base filename
        if (origBase === foundBase) {
            result += origBase;
        } else {
            result += '<span style="color: orange;">' + foundBase + '</span>';
        }
        
        // Extension
        if (foundExt) {
            if (origExt === foundExt) {
                result += '.' + foundExt;
            } else {
                result += '<span style="color: green;">.' + foundExt + '</span>';
            }
        }
        
        return result;
    },

    getBaseName: function(fileName) {
        var lastDot = fileName.lastIndexOf('.');
        return lastDot > 0 ? fileName.substring(0, lastDot) : fileName;
    },

    getExtension: function(fileName) {
        var lastDot = fileName.lastIndexOf('.');
        return lastDot > 0 ? fileName.substring(lastDot + 1) : '';
    },

    onArchiveCellClick: function(view, td, cellIndex, record, tr, rowIndex, e) {
        var target = e.getTarget();
        if (target && target.hasAttribute('data-action')) {
            var action = target.getAttribute('data-action');
            switch(action) {
                case 'open-tree-browser':
                    store_tree.currentPage = 1;
                    store_tree.proxy.extraParams = {
                        caller: 'changeDir',
                        procedure: 'changeParent',
                        node: record.get('elementUrl')
                    };
                    store_tree.load();
                    break;
                
                case 'show-file-details':
                    var elementUrl = record.get('elementUrl');
                    var userappwin = null;
                    var node = encodeURIComponent(elementUrl);
                    var url = document.hammerForm.userapp.value + '?urlcaller=restore&modus=search&elementurl=' + node
                    userappwin = window.open(url,"userapp","height=750,width=620,scrollbars=auto,resizable,status");
                    userappwin.focus();
                    break;
            }
        }
    },

    onShowTroubleshootingHelp: function() {
        P5Index.view.TroubleshootingHelp.show();
    },

    onSendFeedback: function() {
        var subject = encodeURIComponent('Media Recovery Parser Issue');
        var machineId = encodeURIComponent(window.location.hostname || 'Unknown');
        var body = encodeURIComponent(
            'Machine ID: ' + (window.location.hostname || 'Unknown') + '\n' +
            'Browser: ' + navigator.userAgent + '\n\n' +
            'Project File Format: [Please specify: EDL/OTIO/FCPXML/ALE/etc.]\n' +
            'Expected number of files: \n' +
            'Actual number of files found: \n' +
            'Additional details: \n\n' +
            'Please attach your project file to this email.\n'
        );
        window.location = 'mailto:feedback@archiware.com?subject=' + subject + '&body=' + body;
    },

    onLoggingToggle: function(checkbox, newValue) {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        this.loggingEnabled = newValue;
        
        if (!newValue) {
            this.setDebugMode(false);
            logger.stopLogging();  // NEU: Use logger directly
            
            // NEU: Log deactivation
            if (logger.isLogging) {
                logger.addLog('Logging deactivated');
            }
        }
    },

    onDebugModeToggle: function(checkbox) {
        // Nur wenn Logging aktiv ist
        if (checkbox.getValue()) {
            this.setDebugMode(true);
        }
    },

    setDebugMode: function(enabled) {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        this.debugMode = enabled;
        
        // Update UI
        var checkbox = this.lookupReference('enableLogging');
        checkbox.setBoxLabel(enabled ? 'Log (Debug)' : 'Log');
        
        // Update alle Parser
        var parsers = ['OTIOParser', 'EDLParser', 'ALEParser', 'XMEMLParser', 'TEXTParser', 'TSVParser', 'FCPXMLParser'];
        parsers.forEach(function(parserName) {
            if (P5Index.util[parserName]) {
                P5Index.util[parserName].debugMode = enabled;
            }
        });
        
        // NEU: Update OptimizedMediaParser und MediaCustomerFilter
        if (P5Index.util.OptimizedMediaParser) {
            P5Index.util.OptimizedMediaParser.debugMode = enabled;
        }
        if (P5Index.util.MediaCustomerFilter) {
            P5Index.util.MediaCustomerFilter.debugMode = enabled;
        }
        
        // NEU: Log debug mode change
        if (logger.isLogging) {
            logger.addLog('Debug mode: ' + (enabled ? 'ENABLED' : 'DISABLED'));
        }
        
        console.log('Debug mode:', enabled ? 'ON' : 'OFF');
    },

    // Show Log Window
    onShowLogWindow: function() {
        Ext.create('P5Index.view.ParserLogWindow', {
            viewModel: this.getViewModel()  // Teilt das ViewModel
        }).show();
    },

    formatFileSize: function(bytes) {
        if (bytes < 1024) {
            return bytes + ' Bytes';
        } else if (bytes < 1024 * 1024) {
            return (bytes / 1024).toFixed(1) + ' KB';
        } else if (bytes < 1024 * 1024 * 1024) {
            return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
        } else {
            return (bytes / (1024 * 1024 * 1024)).toFixed(2) + ' GB';
        }
    }
});