
##############################################################################
#
# login.tcl --
#
#    Registers procedures for application login and command dispatch.
#    For all modules tracked by the login module, this code registers
#    three url handlers:
#
#      <context>         - command dispatcher
#      <context>/login   - login handler
#      /proxy            - proxy handler
#
#
#    The "<context>" is the url-prefix for a particular module.
#    Each module should have unique prefix within the system.
#    The module must register itself within the login module.
#    For the registration, the context, namespace and the start
#    page/procedure name are required (see "register" procedure).
#
#    The login module does initialize itself within itself as well.
#    This is needed so login tdp pages can access their resources
#    in the same way as other modules do.
#
#    This file generates following global resources:
#
#      o. one "logincfg" sv_array
#
#    See the file "license.txt" for information on usage and
#    redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
#    Rcsid: @(#)$Id: login.tcl,v 1.277 2019/01/14 11:30:47 zv Exp $
#
##############################################################################

##############################################################################
#
#    Initialize module namespace.
#    This code gets executed once, in the startup thread.
#
#    NOTE: caller should be loading us in the correct module namespace.
#
##############################################################################

namespace eval [namespace current] {

    variable modenv

    #
    # Following module configuration parameters are supported
    #
    #  developFlag - source all module tcl files for each request.
    #  customPages - turn-on extra filesystem lookup for appl resources
    #

    set c "ns/server/[ns_info server]/module/$modenv(NAME)"

    variable C

    foreach opt {
        {developFlag "-bool" 0} {customPages "-bool" 0}
    } {
        foreach {o t d} $opt {
            set C($o) [eval ns_config $t $c $o "$d"]
            unset o t d
        }
    }

    #
    # The context of this (login) module.
    #

    variable LoginContext $modenv(URL)

    #
    # Setup document search path depending on wether
    # user has overriding supplied appl pages or not.
    #
    # If yes, the path where we search pages and other
    # application resources (javascript libs, ccs files,
    # images, html docs, etc) can be rather complicated.
    #
    # If not, we just serve static pages from nsx cache
    # and look for pages/procedures in the interpreter.
    #

    variable SearchPath           \
        [list                     \
             [ns_server pagedir]  \
             [ns_library private] \
             [ns_library shared]  \
            ]

    #
    # Registered module database array
    #

    variable GC $modenv(NAME)cfg

    #
    # Register the proxy handlers
    #

    ns_register_proc GET  /proxy [namespace current]::proxy
    ns_register_proc POST /proxy [namespace current]::proxy
    ns_register_proc HEAD /proxy [namespace current]::proxy

    unset opt c
}

##############################################################################
#
#                         Public module procedures
#
##############################################################################

##############################################################################
#
# login --
#
#    Validates user credentials (username/password). It is activated for
#    both GET and POST http methods.
#
#    On POST, it expects two form fields:
#
#        uname - holding username (expects verbatim: [encrypted])
#        upass - holding the RSA-encrypted authentication key
#
#    Additionally, it MAY auto-login non-root users.
#
#    On GET method:
#       - displays the login form page OR auto-logs the user
#
#    On POST and validation
#       - failure: redirect to itself with message selector in url.
#       - success: create new session and redirect to start page
#
# Arguments:
#    context: URL context of the module being processed (/lexxapp, login)
#
# Results:
#    None.
#
##############################################################################

proc login {context} {

    variable C
    variable GC
    variable LoginContext

    variable Namespace  [namespace current]
    variable Context    $LoginContext
    variable AutoLogin  0
    variable AutoUser   ""
    variable ErrorCause 1

    set myself [ns_conn url]?[ns_conn query]
    set method [ns_conn method]

    #
    # If running under login module context, take the first of the modules
    # as listed in the registered modules list to jump into.
    #

    if {$context ne $LoginContext} {
        set ctx $context
    } else {
        set ctx [lindex [modlist] 0]
    }

    #
    # Referer may have been set by the dispatch procedure
    # which decided to redirect to us here in order to
    # authenticate the user.
    #

    set referer [readCookie Referer]
    set sid [CookieSession]

    if {$sid > 0 && [session::ctrl::valid $sid]} {

        #
        # Shortcut: if we got fed with the request that already
        # contains the valid session, try that one.
        #

        if {$referer ne {}} {
            set modstart $referer
        } else {
            set modstart [modstart $ctx]
        }

        if {[ns_conn query] ne {}} {
            append modstart [expr {[regexp {\?} $modstart] ? "&" : "?"}]
            append modstart [ns_conn query]
        }

        Redirect $modstart $sid

    } elseif {$method eq {POST}} {

        #
        # Looks like something like our own login
        # page (or something that emulates it)
        # is calling us.
        #

        # This post condition is passed twice.
        # These two passes are necessary to implement 2FA but also to remain
        # compatible with older logins.
        # The first time initiated by an Ajax call that validates the credentials.
        # The second time by form-submit which performs the login.
        # This is controlled by the caller variable. With Ajax $caller = "validate"

        set qset [ns_conn form]

        set caller [ns_set get $qset caller]
        set user   [ns_set get $qset uname]
        set pass   [ns_set get $qset upass]
        set mfa    [ns_set get $qset umfa]
        set token  [ns_set get $qset ucode]

        #ns_log notice caller:$caller
        #ns_log notice user1:$user
        #ns_log notice pass1:$pass
        #ns_log notice mfa:$mfa
        #ns_log notice token:$token

        #
        # Decrypt username, password, login-key
        #

        set loginkey ""

        if {$user eq {} || $user eq {[encrypted]}} {
            if {[catch {rsa::decode $pass} upenc]} {
                set pip [ns_conn peeraddr]
                ns_log warning "Bad login: ($pip) invalid authentication: $pass"
                return [Redirect $myself]
            }
            lassign [split $upenc :] user nonce pass loginkey
            if {$caller ne "validate"} {
                if {$nonce ne {} && ![utility::noonce $nonce]} {
                    set pip [ns_conn peeraddr]
                    ns_log warning "Bad login: ($pip) invalid nonce for user: $user"
                    return [Redirect $myself]
                }
            }
            if {$loginkey ne {}} {
                lassign [split $loginkey -] rsrc lkey
                if {$rsrc ne {Workstation}} {
                    set pass $pass:$rsrc
                    set loginkey ""
                } else {
                    set loginkey $lkey
                }
            }
        }

        #
        # See if username or login-key based login
        #

        if {$loginkey eq {}} {

            #
            # User-based
            #

            set login_msg  ""
            set login_next ""
            set login_okay 1
            # login_okay   0 return error
            #              1 return okay

            set login_code 1
            # login_code   0 page status error

            set usergrp [UserPrefGroup new -volatile]

            if {[UserPrefGroup names $user] eq {}} {
                set pip [ns_conn peeraddr]
                set login_okay 0
                set login_msg  "Bad login: ($pip) unknown user: $user"
                ns_log warning $login_msg
            } else {
                if {![UserPref $user checkpass $pass]} {
                    set pip [ns_conn peeraddr]
                    set login_okay 0
                    set login_msg  "Bad login: ($pip) invalid password for user: $user"
                    ns_log warning $login_msg
                }
                if { $mfa == 1 && ![UserPref $user checktoken $token]} {
                    set pip [ns_conn peeraddr]
                    set login_okay 0
                    set login_msg  "Bad login: ($pip) invalid 2fa token for user: $user"
                    ns_log warning $login_msg
                }
            }

            if {$caller eq "validate"} {

                if {$login_okay} {

                    # check if MFA is enabled
                    set forceMFA 0
                    set checkMFA 0
                    set newAgent 0
                    set isExired 0
                    set res [SystemConfigGroup new -volatile]
                    if { [catch {dict get [[$res "GUI"] asdict] modul.Login.multi-factor-authentication.configurable.boolean} checkMFA] } {
                        set forceMFA 1
                        set login_msg    "MFA validation: System configuration for multi-factor authentication could not be determined."
                        ns_log warning $login_msg
                    }

                    # check if the user has already logged in with mfa from this browser
                    if { $checkMFA && ![UserPref $user checkfingerprint [AgentFingerprint]]} {
                        set newAgent 1
                        set login_msg "MFA validation: Invalid browser fingerprint for user: $user"
                        ns_log warning $login_msg
                    }

                    # check if the MFA has to be renewed by the user
                    if { $checkMFA && [UserPref $user isexpired]} {
                        set isExired 1
                        set login_msg "MFA validation: MFA must be renewed by the user: $user"
                        ns_log warning $login_msg
                    }

                    # MFA is not invoked if:
                    # - the user has previously logged in through this browser.

                    if { $forceMFA || ($checkMFA && $newAgent) || ($checkMFA && ![CookieSession]) || ($checkMFA && $isExired) } {
                        UserPref $user sendtoken
                        set login_next "mfa"
                    } else {
                        set login_next "login"
                    }

                }

                set x [yajl create \#auto]
                $x map_open \
                    map_key success  bool   $login_okay \
                    map_key code     number $login_code \
                    map_key message  string $login_msg \
                    map_key next  string $login_next \
                map_close

                set json_array [$x get]

                ns_return 200 text/html $json_array

                return

            } else {
                if {!$login_okay} {
                    return [Redirect $myself]
                }
            }

            # When MFA is active and login goes through:
            #   - the fingerprint is stored in the user resource, and
            #   - when the MFA expires, it is renewed
            if { $mfa == 1 } {
                UserPref $user setfingerprint [AgentFingerprint]
                UserPref $user setexpired
            }

        } else {

            #
            # Login-key-based (at the moment only Workstation key)
            #

            if {[WorkstationGroup available $user] < 1} {
                ns_log warning "Bad login: unknown workstation: $user"
                return [Redirect $myself]
            }
            if {[Workstation $user loginkey] ne $loginkey} {
                ns_log warning "Bad login: key mismatch for workstation: $user"
                return [Redirect $myself]
            }
            if {$::tcl_platform(platform) eq {windows}} {
                set user Administrator
            } else {
                set user [const adminUser]
            }
        }

        #
        # Get accepted languages
        #

        set allangs [ns_set iget [ns_conn headers] Accept-Language]
        set deflang [string trim [lindex [split $allangs ,-_] 0]]

        #
        # Create session now that user is authenticated
        #

        set sid [session::ctrl::create]
        session::ctrl::config $sid \
            -usersHome [UserPref $user homedir] \
            -remoteUser $user \
            -remotePass $pass \
            -locale $deflang  \
            -userIsOperator [utility::isOperator $user] \
            -userIsAdmin [utility::isAdministrator $user]

        if {$loginkey eq {}} {

            #
            # Run generic, then per-user login handlers
            #

            if {[sv_get $GC atlogin- script]} {
                if {[catch {uplevel \#0 $script} err]} {
                    ns_log warning "Bad login: atlogin- script: $err"
                    return [Redirect $myself]
                }
            }

            if {[sv_get $GC atlogin-$user script]} {
                if {[catch {uplevel \#0 $script} err]} {
                    ns_log warning "Bad login: atlogin-$user script: $err"
                    return [Redirect $myself]
                }
            }

            #
            # Register per-user, then generic logout handlers.
            # Instead of doing it in [logout] make sure they
            # are run in order when the session gets terminated.
            #

            if {[sv_get $GC atlogout-$user script]} {
                session::ctrl::config $sid -atDestroy $script
            }

            if {[sv_get $GC atlogout- script]} {
                session::ctrl::config $sid -atDestroy $script
            }
        }

        #
        # Attach to newly created session, prepare the
        # module context by running its own atstart
        # callback and redirect to its startup page
        # or to redirection url.
        #

        session::ctrl::pop $sid $ctx

        if {[sv_get $GC atstart-$ctx script]} {
            if {[catch {uplevel \#0 $script} err]} {
                catch {session::ctrl::push $sid 1}
                ns_log warning "Bad login: atstart-$ctx script: $err"
                return [Redirect $myself]
            }
        }

        if {$referer ne {}} {
            set modstart $referer
        } else {
            set modstart [modstart $ctx]
        }

        if {[ns_conn query] ne {}} {
            append modstart [expr {[regexp {\?} $modstart] ? "&" : "?"}]
            append modstart [ns_conn query]
        }

        Redirect $modstart $sid

    } elseif {$method eq {GET}} {

        #
        # For localhost login, user is already "logged-in"
        # so we can save ourselves the login mask if the
        # autoLogin for this user is allowed and the user
        # is not the generic adminUser.
        #

        set user ""
        set qset [ns_conn headers]
        set peer [ns_conn peeraddr]

        if {![regexp {([^:]+):?} [ns_set iget $qset Host] _ addr]} {
            set addr ""
        }

        if {($peer eq {127.1} || $peer eq {127.0.0.1}) && $addr eq {localhost}} {
            if {($::tcl_platform(platform) eq {windows})} {
                if {[catch {getCurrentConsoleUser} user]} {
                    ns_log warning "Bad login: can't get console user: $user"
                    set user ""
                }
            } else {
                if {[catch {file attributes /dev/console -owner} user]} {
                    ns_log warning "Bad login: can't get console user: $user"
                    set user ""
                } elseif {$user eq {root}} {
                    set user ""
                }
            }
        }

        if {$user ne {} && [catch {UserPrefGroup names $user} user]} {
            set user ""
        }

        if {$user eq {} || [ns_set get [ns_conn form] c] == 2} {
            set uauto 0
        } else {
            set uauto [UserPref $user autoLogin]
        }

        if {$uauto && $user ne {} && $user ne [const adminUser]} {

            #
            # Wow, this user may auto-login bypassing
            # the login page entirely. We need to
            # replicate much of the above processing...
            #

            if {[sv_get $GC atlogin- script]} {
                if {[catch {uplevel \#0 $script} err]} {
                    ns_log warning "Bad login: atlogin- script: $err"
                    return [Redirect $myself]
                }
            }

            if {[sv_get $GC atlogin-$user script]} {
                if {[catch {uplevel \#0 $script} err]} {
                    ns_log warning "Bad login: atlogin-$user script: $err"
                    return [Redirect $myself]
                }
            }

            #
            # Get accepted languages
            #

            set allangs [ns_set iget [ns_conn headers] Accept-Language]
            set deflang [string trim [lindex [split $allangs ,-_] 0]]

            set sid [session::ctrl::create]
            session::ctrl::config $sid \
                -usersHome [UserPref $user homedir] \
                -remoteUser $user \
                -remotePass [UserPref $user password] \
                -locale $deflang  \
                -userIsOperator [utility::isOperator $user] \
                -userIsAdmin [utility::isAdministrator $user]

            if {[sv_get $GC atlogout-$user script]} {
                session::ctrl::config $sid -atDestroy $script
            }

            if {[sv_get $GC atlogout- script]} {
                session::ctrl::config $sid -atDestroy $script
            }

            session::ctrl::pop $sid $ctx

            if {[sv_get $GC atstart-$ctx script]} {
                if {[catch {uplevel \#0 $script} err]} {
                    ns_log warning "Bad login: atstart-$ctx script: $err"
                    catch {session::ctrl::push $sid 1}
                    return [Redirect $myself]
                }

            }

            if {$referer ne {}} {
                set modstart $referer
            } else {
                set modstart [modstart $ctx]
            }

            if {[ns_conn query] ne {}} {
                append modstart [expr {[regexp {\?} $modstart] ? "&" : "?"}]
                append modstart [ns_conn query]
            }

            Redirect $modstart $sid

        } else {

            #
            # Not auto-login; display login page.
            #

            variable AutoUser ""
            variable AutoLogin 0

            if {$user ne {} && ![utility::isAdministrator $user]} {
                variable AutoUser $user
                variable AutoLogin $uauto
            }

            switch -- [sendpage login.tdp] {
               -1 {SendErrorPage}
                1 {ns_returnnotfound}
            }
        }
    }
}

##############################################################################
#
# register --
#
#    Register an application within the login module.
#
# Arguments:
#    ctx        context (url-prefix) where the application resides
#    nso        namespace-absolute name of the startup procedure
#    args       optional; key/value pairs
#
# Results:
#    None.
#
##############################################################################

proc register {ctx nsp args} {

    variable GC
    variable LoginContext

    #
    # Load optional arguments
    #

    set opts(-indexpage) index.tdp
    set opts(-errorpage) ""

    array set opts $args

    set ns [namespace current]

    #
    # Register the module namespace
    #

    sv_set $GC namespace-$ctx $nsp

    #
    # Register the module-wide procedures
    #

    ns_register_proc GET  $ctx/login ${ns}::login $ctx
    ns_register_proc POST $ctx/login ${ns}::login $ctx

    ns_register_proc GET  $ctx/logout ${ns}::logout $ctx
    ns_register_proc POST $ctx/logout ${ns}::logout $ctx

    #
    # Register the url dispatcher procedure
    #

    ns_register_proc GET  $ctx ${ns}::dispatch $ctx
    ns_register_proc POST $ctx ${ns}::dispatch $ctx

    #
    # Updates login database.
    #

    foreach opt [array names opts] {
        sv_set $GC $opt-$ctx $opts($opt)
    }

    foreach opt {prepage postpage atstart} {
        sv_set $GC $opt-$ctx ""
    }

    if {$ctx ne $LoginContext} {

        #
        # The login module is *not*
        # reported as regular module.
        #

        sv_lappend $GC modlist $ctx
    }

    return $ctx
}

##############################################################################
#
# modlist --
#
#    List names of all registered modules.
#
# Arguments:
#    None
#
# Results:
#    List of module contexts. Each module context is actually
#    the server-relative part of the module url..
#
##############################################################################

proc modlist {} {

    variable GC
    sv_get $GC modlist
}

##############################################################################
#
# modinfo --
#
#    Interrogates module options.
#
# Arguments:
#    None
#
# Results:
#    Option value(s)
#
##############################################################################

proc modinfo {context option} {

    variable GC

    if {[string index $option 0] == "-"} {
        set what "$option-$context"
    } else {
        set what "-$option-$context"
    }
    if {[sv_exists $GC $what]} {
        sv_get $GC $what
    }
}

##############################################################################
#
# modstart --
#
#    Returns bootstrap-url of the module. Needs active valid session.
#
# Arguments:
#    None
#
# Results:
#    Option value(s)
#
##############################################################################

proc modstart {context} {

    if {[set sid [session::id]] == 0} {
        return $context
    } else {
        return $context/$sid ; # FIXME: dependency on the session::url !!!
    }
}

##############################################################################
#
# atstart --
#
# Arguments:
#    context    module url context
#    script     script to be executed
#
# Results:
#    None.
#
##############################################################################

proc atstart {context script} {

    variable GC

    set nsp [namespace qualifier $script]
    if {$nsp eq {}} {
        set script [sv_get $GC namespace-$context]::$script
    }

    sv_set $GC atstart-$context $script
}

##############################################################################
#
# prepage --
#
#    Register an pre-page handler. The handler is entered in list of
#    pre-handlers. These handlers are activated before the application
#    page is executed.
#
# Arguments:
#    context    context (url-prefix) where the application resides
#    script     script to be executed
#    page       string match of the page name (default = matches all pages)
#
# Results:
#    Handler's handle in the global database array.
#
##############################################################################

proc prepage {context script {page "*"}} {

    variable GC
    set key prepage-$context

    set nsp [namespace qualifier $script]
    if {$nsp == ""} {
        set script [sv_get $GC namespace-$context]::$script
    }

    sv_lappend $GC $key [list $script $page]

    return $key:[lsearch [sv_get $GC $key] $key]
}

##############################################################################
#
# postpage --
#
#    Register an post-page handler. The handler is entered in list of
#    post-handlers. These handlers are activated after the application
#    page is executed.
#
# Arguments:
#    context    context (url-prefix) where the application resides
#    script     script to be executed
#    page       string match of the page name (default = matches all pages)
#
# Results:
#    Handler's handle in the global database array.
#
##############################################################################

proc postpage {context script {page "*"}} {

    variable GC
    set key postpage-$context

    set nsp [namespace qualifier $script]
    if {$nsp == ""} {
        set script [sv_get $GC namespace-$context]::$script
    }

    sv_lappend $GC $key [list $script $page]

    return $key:[lsearch [sv_get $GC $key] $key]
}

##############################################################################
#
# atlogin --
#
# Arguments:
#    user       name of the user
#    script     script to be executed
#
# Results:
#    None.
#
##############################################################################

proc atlogin {script {user ""}} {

    variable GC
    sv_set $GC atlogin-$user $script
}

##############################################################################
#
# atlogout --
#
# Arguments:
#    user       name of the user
#    script     script to be executed
#
# Results:
#    None.
#
##############################################################################

proc atlogout {script {user ""}} {

    variable GC
    sv_set $GC atlogout-$user $script
}

##############################################################################
#
# cancel --
#
#    Cancels some registered prepage/postpage handler
#
# Arguments:
#    handle     handler's handle :)
#
# Results:
#    None.
#
##############################################################################

proc cancel {handle} {

    variable GC

    set h [split $handle ":"]

    set k [lindex $h 0]
    set x [lindex $h 1]

    if {[sv_exist $GC $k] && [string is integer -strict $x]} {
        catch {
            # FIXME: potential race condition
            sv_set $GC $k [lreplace [sv_get $GC $k] $index $x]
        }
    }
}

##############################################################################
#
# valid --
#
#    Checks wether a valid login exists.
#    It also refreshes the session idle timer.
#
# Arguments:
#    None.
#
# Results:
#    0 - no valid login
#   >0 - valid login
#
##############################################################################

proc valid {context} {

    set sid [session::ctrl::pop [ns_conn url] $context]

    if {$sid > 0} {
        session::ctrl::push $sid
    }

    return $sid
}

##############################################################################
#
# logout --
#
#    Logs out the given session and redirects to login module start page
#  . It is activated for both GET and POST http methods.
#
# Arguments:
#    Module context
#
# Results:
#    Always redirects the browser.
#
##############################################################################

proc logout {context} {

    set cid [CookieSession]

    if {$cid > 0 && [session::ctrl::valid $cid]} {
        catch {session::ctrl::push $cid}
    }

    set referer [readCookie Referer]
    CreateCookie Referer ""

    if {$referer ne {}} {
        set modstart $referer
    } else {
        set modstart $context/login
    }

    if {   [ns_set get [ns_conn form] sid] ne {}
        && [string first "c=2" $referer] == -1} {

        #
        # Turn off auto-logins when somebody logs out
        # so the next will need to login explicitly.
        #

        append modstart [expr {[regexp {\?} $modstart] ? "&" : "?"}]
        append modstart c=2 ; # Turn off auto-login
    }

    Redirect $modstart
}

##############################################################################
#
# procpage --
#
#    Returns the current running page.
#
# Arguments:
#    Optional anything which allows us to return empty value if the
#    local variable Page is not existent.
#
# Results:
#    None.
#
##############################################################################

proc procpage {{var ""}} {

    variable Page

    if {$var == "" || [info exists Page]} {
        return $Page
    }
}

##############################################################################
#
# pageproc --
#
#    Returns the name of the Tcl procedure containing the current page
#
# Arguments:
#    Optional anything which allows us to return empty value if the
#    local variable PageProc is not existent.
#
# Results:
#    None.
#
##############################################################################

proc pageproc {{var ""}} {

    variable PageProc

    if {$var == "" || [info exists PageProc]} {
        return $PageProc
    }
}

##############################################################################
#
# dispatch --
#
#    Dispatches url to its servicing procedure.
#
# Arguments:
#    context    url context under which this command is called
#
# Results:
#    Ignore any results.
#
##############################################################################

proc dispatch {context} {

    variable GC
    variable LoginContext

    variable Context $context
    variable Namespace [sv_get $GC namespace-$context]

    set url [regsub $context/? [ns_conn url] {}]

    if {$url eq {} && [ns_set size [ns_conn form]] == 0} {

        #
        # Called with the empty context.
        #

        CreateCookie Referer ""

        if {$context eq $LoginContext} {
            Redirect [lindex [modlist] 0]/login
        } else {
            Redirect $context/login
        }

        return
    }

    set sid [session::ctrl::pop $url $context page]

    if {$sid < 0} {

        #
        # URL refers to invalid session.
        #

        catch {error "Invalid session in URL"}
        SendErrorPage

        return
    }

    #
    # Initialize currently running page
    #

    if {![info exists page]} {
        if {$sid > 0 || $url eq {}} {
            set page [sv_get $GC -indexpage-$context]
        } else {
            set page [ns_conn url]
        }
    }

    #
    # See if page a [D]ocument or [P]rogram page
    #

    switch -glob $page {
        *.adp - *.tdp {
            set ptype P
        }
        default {
            set ptype D
        }
    }

    #
    # Handle session ID. Valid session ID must be sent
    # either in cookie alone OR both cookie and URL.
    #

    set cid [CookieSession]

    if {$sid > 0 && $sid != $cid} {
        catch {error "Session cookie/URL mismatch - Cookie: $cid | URL/Session: $sid | Page: $page"}
        SendErrorPage
        return
    }

    if {$sid == 0 && $cid > 0} {
        set sid [session::ctrl::pop $cid $context]
    }

    if {$sid <= 0 && $ptype eq {P} && $context != $LoginContext} {

        #
        # On invalid session bail out.
        #

        if {$sid < 0 && ($url ne {} || [ns_set size [ns_conn form]] == 0)} {
            catch {error "Invalid session in cookie"}
            SendErrorPage
            return
        }

        #
        # All modules pages, except for the login module
        # must be authenticated and have a valid session.
        # Also make sure to set the referer if we are
        # handling special urls that consist only of the
        # context and the query.
        #

        if {$url eq {} && [ns_set size [ns_conn form]] > 0} {
            if {[ns_conn query] eq {}} {
                CreateCookie Referer [ns_conn url]
            } else {
                CreateCookie Referer [ns_conn url]?[ns_conn query]
            }
        }

        #
        # Pass the whole query to login procedure as it
        # as contain some keys that might influence it.
        #

        if {[ns_conn query] eq {}} {
            Redirect $context/login
        } else {
            Redirect $context/login?[ns_conn query]
        }

    } else {

        #
        # Run page or send a static document.
        # For program pages there must be a
        # valid session ID. For document pages
        # the session is not required.
        #

        variable C

        session::ctrl::proxyprefix $sid [ProxyPrefix]

        if {$sid > 0} {
            CreateCookie Session [SessionCookie $sid] ; # [session::expires]
        }

        switch $ptype {
            P {
                if {$C(customPages) && $C(developFlag)} {
                    RefreshLibs
                }
                set stat [sendpage $page]
                if {$stat == 0} {
                    session::ctrl::push $sid
                } else {
                    # catch {session::ctrl::push $sid 1}
                    catch {session::ctrl::push $sid}
                    if {$stat == 1} {
                        catch {error "no such page: $context/$page"}
                    }
                    SendErrorPage
                }
            }
            D {
                set stat [senddoc $page]
                if {$stat == 0} {
                    if {$sid > 0} {
                        session::ctrl::push $sid
                    }
                } else {
                    if {$sid > 0} {
                        # catch {session::ctrl::push $sid 1}
                        catch {session::ctrl::push $sid}
                    }
                    if {$stat == 1} {
                        ns_log error "no such document: $page"
                    } else {
                        ns_log error "error sending document: $page"
                    }
                }
            }
        }
    }
}

##############################################################################
#
# proxy --
#
#    Handles proxy requests.
#
# Arguments:
#    None.
#
# Results:
#    Ignore any results.
#
##############################################################################

proc proxy {args} {

    #
    # Format of url that we understand is:
    #
    #    http://host:port/proxy/server/servername/my/page
    #    http://host:port/proxy/client/clientname/my/page
    #
    # We will rewrite the request and replace host:port with new
    # hostname and port of the server (or client) given in the
    # request. The request URL will be rewritten to strip first
    # three elements so the new will look like:
    #
    #    http://newhost:newport/my/page
    #
    # This URL will be fetched and results forwarded to the
    # request initiator.
    #

    set part [split [ns_conn url] /]
    set kind [string totitle [lindex $part 2]]
    set what [lindex $part 3]

    #
    # Get new hostname and port using URL specified server
    # or client record.
    #

    set host ""
    set port ""

    switch $kind {
        Client - Server {
            catch {
                set srvc [$kind $what service]
                if {$srvc eq {}} {
                    set host [$kind $what hostname]
                    set port [$kind $what port]
                } else {
                    set dmn [$kind $what domain]
                    lassign [DnsSd resolve$kind $srvc $dmn] host port
                }
            }
        }
    }

    if {$host eq {}} {
        return [ns_returnnotfound]
    }

    #
    # Append rewritten URL
    #

    append url http://$host:$port
    append url / [join [lrange $part 4 end] /]

    set query [ns_conn query]
    if {$query ne {}} {
        append url ? $query
    }

    #
    # Make sure remote URL writing code rewrites
    # all URL's back to proxy.
    #

    ns_set cput [ns_conn headers] ProxyPrefix [join [lrange $part 0 3] /]

    #
    # Now run the proxy.
    #

    if {[catch {ns_proxy_handler_http $url 30} err]} {
        ns_log error "proxy: '$err'"
    }
}

##############################################################################
#
# sendpage --
#
#    Runs tdp page from the file-system or from library or from a cached
#    procedure. It runs any registered pre- and/or post-page handlers
#    before/after running the application page.
#
# Arguments:
#    page        name of page/procedure (in namespace-relative form)
#
# Results:
#    0 - all went fine
#    1 - no page found
#   -1 - page error
#
##############################################################################

proc sendpage {page} {

    variable C
    variable Context
    variable Recursion
    variable Namespace

    #
    # First, try finding the page in the file-system if allowed
    # by the customization
    #

    utility::killcache

    SetLocale

    if {$C(customPages)} {
        variable SearchPath
        foreach dir $SearchPath {
            set ppage [ns_normalizepath $dir/$Context/$page]
            if {[file readable $ppage]} {
                switch -glob $page {
                    *.adp {set cmd [list SendAdp $ppage]}
                    *.tdp {set cmd [list tdp::runpage $ppage $Namespace]}
                }
                break
            }
        }
    }

    #
    # Next, try finding the page in the cache
    #

    if {![info exists cmd]} {
        switch -glob $page {
            *.adp {
                set ppage [ns_normalizepath $Context/$page]
                set cmd [list CacheAdp $Context $page]
            }
            *.tdp {
                set ppage ${Namespace}::$page
                set cmd [list tdp::runcmd $ppage]
            }
            default {
                error "unsupported page type: $page"
            }
        }
    }

    #
    # Set's so pre/post page handlers
    # can find-out where they are...
    #

    variable Page $page
    variable PageProc $ppage

    if {[info exists Recursion] && $Recursion > 0} {
        session::ctrl::GetParent $page
        set ret [eval $cmd]
        if {$ret == -1} {
            ns_log warning "page/procedure $ppage ran with error"
        }
        return $ret
    }

    set Recursion 1

    switch -- [RunHandlers prepage] {
        "0" - "3" - "4" {
            #
            # 0=TCL_OK, 3=TCL_BREAK or 4=TCL_CONTINUE
            #
            set ret [eval $cmd]
            if {[RunHandlers postpage] > 0} {
                ns_log warning "postpage handler exception."
            }
            if {$ret == -1} {
                ns_log warning "page/procedure $ppage ran with error"
            }
        }
        "2" {
            #
            # 2=TCL_RETURN
            #
            ns_log notice "prepage handler signalized return; page skipped."
            set ret 0
        }
        default {
            #
            # unknown, assume TCL_ERROR
            #
            ns_log warning "prepage handler signalized error; page skipped."
            set ret -1
        }
    }

    set Recursion 0

    return $ret
}

##############################################################################
#
# sendoc --
#
#    Sends some static document content using custom nsx handler.
#    It tries to find it in several places before giving up.
#
#    NOTE: do not send anything to the connection after this one.
#
# Arguments:
#    url        document's url (absolute, starting from pagedir)
#
# Results:
#    0 - document returned to user
#    1 - no document found, nothing returned
#   -1 - error sending document, nothing returned
#
##############################################################################

proc senddoc {url} {

    variable C
    variable Context
    variable SearchPath

    if {$C(customPages) == 0} {

        #
        # Serve document from the cache.
        #

        set stat [nsx returnurl $url -path ""]
        if {$stat != 1} {
            return $stat
        }

        #
        # Nothing in the cache, atempt load from zipfile.
        #

        nsx lockcache
        set found [GetFromZip $Context [regsub $Context/? $url {}] content]
        if {$found == 1} {
            nsx loadcache $url $content
        }
        nsx unlockcache

        if {$found == 1} {
            return [nsx returnurl $url -path ""]
        }
    }

    #
    # Either we found nothing in the cache nor in
    # the zip file, or the customPages is set, so
    # just go to filesystem to look for the page.
    #

    nsx returnurl $url -path $SearchPath
}

##############################################################################
#
#
#                         Private module procedures
#
#       *** Should never be called from outside of this package ***
#
#
##############################################################################

##############################################################################
#
# SendErrorPage --
#
#    Runs standard error tdp page from the file-system or from library
#    or from cached procedure.
#
# Arguments:
#    context     module url context
#    ns          current module namespace
#
# Results:
#    None.
#
##############################################################################

proc SendErrorPage {} {

    variable C
    variable GC

    variable Context
    variable Namespace

    #
    # Get error-page name. Use default
    # if user has not registered one.
    #

    set page [sv_get $GC -errorpage-$Context]

    if {[string length $page] == 0} {
        variable LoginContext
        set Context $LoginContext
        set Namespace [namespace current]
        set page error.tdp
    }

    #
    # Look-up page in filesystem (couple of places)
    #

    if {$C(customPages)} {
        variable SearchPath
        foreach dir $SearchPath {
            set ppage [ns_normalizepath $dir/$Context/$page]
            if {[file readable $ppage]} {
                return [tdp::runpage $ppage $Namespace]
            }
        }
    }

    #
    # Run page (proc) from running interpreter
    #

    set ppage ${Namespace}::$page
    tdp::runcmd $ppage
}

##############################################################################
#
# RunHandlers --
#
#    Runs pre or post handlers. Handlers may throw error to signalize
#    some error condition. All other cases are considered ok.
#
# Arguments:
#    type        type of handler (see register procedure for avail. types)
#
# Results:
#    0 - all handlers ran ok
#   >0 - one of the handlers signalized exception
#
##############################################################################

proc RunHandlers {type} {

    variable GC
    variable Page
    variable Context

    foreach script [lindex [sv_array get $GC $type-$Context] 1] {
        if {[string match [lindex $script 1] $Page]} {
            if {[set code [catch {uplevel \#0 [lindex $script 0]} err]]} {
                return $code
            }
        }
    }

    return 0
}

##############################################################################
#
# GetFromZip --
#
#    Get content from zipfile(s). The zipfile(s) are searched relative
#    to the current context in various places arround as defined by
#    the SearchPath variable.
#
# Arguments:
#    ctx:   the current context (e.g. /lexxapp)
#    rurl:  relative url from the context (e.g. some/page.ext)
#    cont:  name of variable to return the content if found
#
# Results:
#    1 - content is found (retuned in the cont variable)
#    0 - content not found
#
##############################################################################

proc GetFromZip {ctx rurl cont} {

    variable SearchPath
    upvar $cont content

    set found 0

    foreach dir $SearchPath {
        set p [ns_normalizepath $dir/$ctx/init.zip]
        if {[catch {zipread [open $p]} zf] == 0} {
            if {[$zf directory $rurl] != ""} {
                set content [$zf read $rurl]
                $zf destroy
                set found 1
                break
            }
            $zf destroy
        }
    }

    return $found
}


##############################################################################
#
# SetLocale --
#
#    Sets the locale as requested by the user.
#
# Arguments:
#    None.
#
# Results:
#    None.
#
##############################################################################

proc SetLocale {} {

    set langs [ns_set iget [ns_conn headers] Accept-Language]

    if {[string length $langs]} {
        msgcat::mclocale [lindex [split $langs ,] 0]
    } else {
        msgcat::mclocale "C"
    }
}

##############################################################################
#
# RefreshLibs --
#
#    Reloads the module library files if in development mode.
#
# Arguments:
#
# Results:
#    None.
#
##############################################################################

proc RefreshLibs {} {

    variable Context
    variable Namespace
    variable SearchPath

    foreach dir $SearchPath {
        set match [ns_normalizepath $dir/$Context/lib/*.*tcl]
        foreach lib [glob -nocomplain $match] {
            if {[catch {
                namespace eval $Namespace [subst {
                    list source $lib
                }]
            } err]} {
                ns_log error "sourcing $lib: $err\n$::errorInfo"
            }
        }
    }
}

##############################################################################
#
# ProxyPrefix --
#
#    Obtains the value of our custom header: ProxyPrefix
#
# Arguments:
#    None.
#
# Results:
#    The prefix string or empty.
#
##############################################################################

proc ProxyPrefix {} {
    ns_set iget [ns_conn headers] ProxyPrefix
}

##############################################################################
#
# Redirect --
#
#    Redirects to some other URL. If a session ID is given, we have
#    been successfully logged-in. In this case remember the encoded
#    session ID in the cookie.
#
# Arguments:
#    where      the url where to redirect
#    sid        optional session ID
#
# Results:
#    None.
#
# Side Effects:
#    If remote peer sent ProxyPrefix, prefix the location.
#
##############################################################################

proc Redirect {to {sid 0}} {

    if {$sid > 0} {
        CreateCookie Session [SessionCookie $sid] ; # [session::expires]
    }

    utility::redirect [ns_normalizepath [ProxyPrefix]/$to] [ns_conn headers]
}

##############################################################################
#
# SendAdp --
#
#    Wrapper to send adp files
#
# Arguments:
#    page      absolute path of the file with adp constructs
#
# Results:
#    0 - sent ok
#   -1 - error in parsing adp
#
##############################################################################

proc SendAdp {page} {

    if {[catch {ns_adp_parse -file $page} out]} {
        ns_log error "$page: $out"
        return -1
    } else {
        ns_return 200 text/html $out
        return 0
    }
}

##############################################################################
#
# CacheAdp --
#
#    Wrapper to send cached adp files. At the moment, no caching
#    is employed. We simply revert to zipfile(s) lookup.
#
#    TODO: add some caching so we need not go to zipfiles always.
#
# Arguments:
#    ctx:    current context
#    page:   context relative path of the file with adp constructs
#
# Results:
#    0 - sent ok
#    1 - page not found
#   -1 - error in parsing adp
#
##############################################################################

proc CacheAdp {ctx page} {

    if {[GetFromZip $ctx $page adp]} {
        if {[catch {ns_adp_parse $adp} out]} {
            ns_log error "$page: $out"
            return -1
        } else {
            ns_return 200 text/html $out
            return 0
        }
    } else {
        return 1
    }
}

##############################################################################
#
# readCookie --
#
#    Returns the current cookie value
#
# Arguments:
#    None.
#
# Results:
#    None.
#
##############################################################################

proc readCookie {name} {

    set n [CookieName $name]
    set q [ns_conn headers]

    set a($n) ""

    foreach p [split [ns_set iget $q Cookie] ";"] {
        if {[regexp {^([^=]+)=(.*)$} [string trim $p] _ key val]} {
            set a($key) $val
        }
    }

    return $a($n)
}

##############################################################################
#
# CreateCookie --
#
#    Create a cookie
#
# Arguments:
#    None.
#
# Results:
#    None.
#
##############################################################################

proc CreateCookie {name value {expires 0}} {

    set n [CookieName $name]
    set q [ns_conn headers]

    if {$expires > 0 || $value eq {}} {
        if {$value eq {}} {
            set t 0
        } elseif {$expires > 0} {
            set t [expr {$expires - [clock seconds]}]
        }
        ns_set put $q Set-Cookie "$n=$value; Max-Age=$t; Path=/; HttpOnly"
    } else {
        ns_set put $q Set-Cookie "$n=$value; Path=/; HttpOnly"
    }
}

##############################################################################
#
# SessionCookie --
#
#    Create session-ID cookie value.
#
# Arguments:
#    None.
#
# Results:
#    Value to be put into the session-ID cookie.
#
##############################################################################

proc SessionCookie {sid} {

    append v [AgentFingerprint] :
    append v [expr {abs(wide(rand()*999999999999999999))}] :
    append v $sid

    utility::tea_encode $v
}

##############################################################################
#
# CookieSession --
#
#    Obtain the session ID from the session cookie.
#
# Arguments:
#    val:   value of the cookie storing encrypted session ID.
#
# Results:
#    Value of the session-ID or zero on invalid or already checked value.
#
##############################################################################

proc CookieSession {} {

    if {![catch {
        lassign [split [utility::tea_decode [readCookie Session]] :] f r s
    }]} {
        if {[AgentFingerprint] eq $f} {
            return $s
        }
    }

    return 0
}

##############################################################################
#
# AgentFingerprint --
#
#    Tries to make a remote agent fingerprint.
#
# Arguments:
#    None.
#
# Results:
#    Fingerprint value
#
##############################################################################

proc AgentFingerprint {} {

    set h [ns_conn headers]

    append v [ns_set iget $h User-Agent]
    append v [ns_conn peeraddr]

    binary scan [md5c $v] H* fpt

    return $fpt
}

##############################################################################
#
# CookieName --
#
#    Generates proxy-aware cookie name.
#
# Arguments:
#    None.
#
# Results:
#    Difference in milliseconds.
#
##############################################################################

proc CookieName {name} {

    set p [ProxyPrefix]

    if {$p ne {}} {
        return $p:$name
    }

    return $name
}

############################### End of file ##################################
