Ext.define('Dashboard.view.widget.ArchiveIndexUsageController', {
    extend: 'Ext.app.ViewController',
    alias: 'controller.archive-index-usage',
    
    init: function() {
        this.callParent();
        
        var viewModel = this.getViewModel();
        var queriesStore = viewModel.getStore('indexQueries');
        var usageStore = viewModel.getStore('indexUsageData');
        
        // Warten bis beide Stores geladen sind
        this.queriesLoaded = false;
        this.usageLoaded = false;
        
        queriesStore.on('load', function() {
            this.queriesLoaded = true;
            this.checkBothStoresLoaded();
        }, this);
        
        usageStore.on('load', function() {
            this.usageLoaded = true;
            this.checkBothStoresLoaded();
        }, this);
    },
    
    checkBothStoresLoaded: function() {
        if (this.queriesLoaded && this.usageLoaded) {
            this.updateQueryCheckboxes();
            this.buildChartsFromData();
        }
    },
    
    getStateKey: function() {
        return 'archive-index-usage-queries';
    },
    
    getSavedQueries: function() {
        var stateKey = this.getStateKey();
        var savedState = localStorage.getItem(stateKey);
        return savedState ? JSON.parse(savedState) : null;
    },
    
    saveQueries: function(queries) {
        var stateKey = this.getStateKey();
        localStorage.setItem(stateKey, JSON.stringify(queries));
    },
    
    updateQueryCheckboxes: function() {
        var me = this;
        var viewModel = me.getViewModel();
        var queries = viewModel.getStore('indexQueries');
        var checkboxGroup = me.lookupReference('queryCheckboxGroup_ref');
        
        if (!checkboxGroup || !queries) return;
        
        var items = [];
        var selectedQueries = [];
        var savedQueries = me.getSavedQueries();  // Verwende die Methode!
        
        queries.each(function(query) {
            var queryName = query.get('queryName');
            var isChecked;
            
            if (savedQueries !== null) {
                isChecked = savedQueries.indexOf(queryName) !== -1;
            } else {
                isChecked = true;  // Default: alle aktiv
            }
            
            items.push({
                boxLabel: query.get('description'),
                name: 'selectedQueries',
                inputValue: queryName,
                checked: isChecked
            });
            
            if (isChecked) {
                selectedQueries.push(queryName);
            }
        });
        
        checkboxGroup.removeAll();
        if (items.length > 0) {
            checkboxGroup.add(items);
        }
        
        // Beim ersten Mal State speichern
        if (savedQueries === null && selectedQueries.length > 0) {
            me.saveQueries(selectedQueries);  // Verwende die Methode!
        }
        
        me.updateQueryButtonText(selectedQueries.length);
    },
    
    updateQueryButtonText: function(activeCount) {
        var button = this.lookupReference('querySelector_ref');
        if (!button) return;
        
        if (activeCount === 0) {
            button.setText('Select Queries');
        } else if (activeCount === 1) {
            button.setText('1 Query Selected');
        } else {
            button.setText(activeCount + ' Queries Selected');
        }
    },
    
    onQuerySelectionChange: function(checkboxGroup, newValue) {
        if (!newValue) return;
        
        var me = this;
        
        var selectedQueries = Array.isArray(newValue.selectedQueries) ? 
                            newValue.selectedQueries : 
                            (newValue.selectedQueries ? [newValue.selectedQueries] : []);
        
        // State speichern
        me.saveQueries(selectedQueries);
        
        // Button-Text aktualisieren
        me.updateQueryButtonText(selectedQueries.length);
        
        // Charts neu bauen
        me.buildChartsFromData();
    },
    
    getActiveChartsCount: function() {
        var savedQueries = this.getSavedQueries();
        return savedQueries ? savedQueries.length : 0;
    },
    
    buildChartsFromData: function() {
        var me = this;
        var view = me.getView();
        var viewModel = me.getViewModel();
        var chartContainer = view.getChartContainer();
        
        var queries = viewModel.getStore('indexQueries');
        var usageData = viewModel.getStore('indexUsageData');
        
        chartContainer.removeAll();
        
        if (!queries || queries.getCount() === 0) {
            chartContainer.add({
                xtype: 'component',
                columnWidth: 1.0,
                html: '<div style="padding: 20px; text-align: center; color: #666;">No queries configured</div>',
                height: 60
            });
            return;
        }
        
        var savedQueries = me.getSavedQueries() || [];
        var activeCharts = 0;
        var chartItems = [];
        var totalActiveCharts = savedQueries.length;
        
        // Für jede aktive Query ein Chart erstellen
        queries.each(function(queryRecord) {
            var queryName = queryRecord.get('queryName');
            
            // Prüfung gegen localStorage
            if (savedQueries.indexOf(queryName) === -1) return;
            
            var description = queryRecord.get('description');
            
            var chartData = [];
            usageData.each(function(dataRecord) {
                if (dataRecord.get('queryName') === queryName) {
                    chartData.push(dataRecord.getData());
                }
            });
            
            if (chartData.length > 0) {
                var chartComponent = view.createTimeSeriesChart(
                    queryName,
                    description, 
                    chartData,
                    activeCharts,
                    totalActiveCharts
                );
                chartItems.push(chartComponent);
                activeCharts++;
            }
        });
        
        if (chartItems.length === 0) {
            chartContainer.add({
                xtype: 'component',
                columnWidth: 1.0,
                html: '<div style="padding: 20px; text-align: center; color: #666;">No queries selected</div>',
                height: 60
            });
        } else {
            chartContainer.add(chartItems);
        }
    },
    
    showError: function(message) {
        var view = this.getView();
        var chartContainer = view.getChartContainer();
        chartContainer.removeAll();
        chartContainer.add({
            xtype: 'component',
            columnWidth: 1.0,
            html: '<div style="padding: 20px; text-align: center; color: #d32f2f;">' + message + '</div>',
            height: 60
        });
    },
    
    reloadData: function() {
        var viewModel = this.getViewModel();
        var period = viewModel.get('selectedPeriod') || 7;
        
        // Reset loaded flags
        this.queriesLoaded = false;
        this.usageLoaded = false;
        
        // Beide Stores neu laden
        var queriesStore = viewModel.getStore('indexQueries');
        var usageStore = viewModel.getStore('indexUsageData');
        
        usageStore.getProxy().setExtraParam('days', period);
        
        queriesStore.reload();
        usageStore.reload();
    },
    
    // Event-Handler
    onPeriodChange: function(segmentedButton, button, pressed) {
        if (pressed) {
            var viewModel = this.getViewModel();
            viewModel.set('selectedPeriod', button.value);
            this.reloadData();
        }
    },
    
    onRefreshData: function() {
        this.reloadData();
        Ext.toast('Data refreshed');
    },

    // Neue Methoden im Controller:

    getColorStateKey: function() {
        return 'archive-index-usage-colors';
    },

    getColorMapping: function() {
        var key = this.getColorStateKey();
        var saved = localStorage.getItem(key);
        return saved ? JSON.parse(saved) : {};
    },

    saveColorMapping: function(mapping) {
        var key = this.getColorStateKey();
        localStorage.setItem(key, JSON.stringify(mapping));
    },

    getColorForQuery: function(queryName) {
        var pastelColors = ['#BFF6FA', '#CDFAA7', '#FEF584', '#FFD5DF', '#EFC6E7', '#ffb366', '#66ffcc', '#ff9999', '#99ccff', '#ffcc99', '#b366ff', '#99ff66', '#ff99ff', '#66b3ff', '#ffb3d9', '#b3ffb3', '#ffffb3', '#d9b3ff', '#ffd9b3', '#b3ffff'];
        var mapping = this.getColorMapping();
        
        if (mapping[queryName]) {
            return mapping[queryName];
        }
        
        // Neue Farbe zuweisen
        var usedColors = Object.values(mapping);
        var availableColors = pastelColors.filter(function(color) {
            return usedColors.indexOf(color) === -1;
        });
        
        // Falls alle Farben verwendet, von vorne beginnen
        if (availableColors.length === 0) {
            availableColors = pastelColors;
        }
        
        mapping[queryName] = availableColors[0];
        this.saveColorMapping(mapping);
        
        return availableColors[0];
    }
});