Ext.define('P5Index.util.OptimizedMediaParser', {
    singleton: true,
    
    debugMode: false,  // NEU: Als Property für externe Steuerung

    /**
     * Einheitlicher Parser mit Customer Filter für alle Formate
     * Verwendet bestehende Parser, fügt aber Customer-Filter hinzu
     * 
     * @param {String} content - Dateiinhalt
     * @param {String} format - Dateiformat (edl, ale, fcpxml, xml, otio, json, txt, etc.)
     * @param {String} fileName - Original Dateiname
     * @param {Object} options - Optionale Einstellungen
     * @returns {Object} Parse-Ergebnis mit gefilterten Media Items
     */
    parseForArchive: function(content, format, fileName, options) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        options = options || {};
        var enableArchiveFilter = options.enableArchiveFilter !== false; // Default: true
        var enableStatistics = options.enableStatistics || false;
        
        // NEU: Log start
        if (logger.isLogging) {
            logger.addLog('OptimizedMediaParser: Processing "' + fileName + '" as ' + format.toUpperCase());
        }
        
        console.log(`🔍 Parsing ${format.toUpperCase()} file: ${fileName}`);
        
        var result;
        var startTime = Date.now();
        
        // BESTEHENDE PARSER UNVERÄNDERT VERWENDEN
        try {
            // NEU: Debug log parser selection
            if (logger.isLogging && debugMode) {
                logger.addLog('Selecting parser for format: ' + format.toLowerCase());
            }
            
            switch (format.toLowerCase()) {
                case 'edl':
                    result = P5Index.util.EDLParser.parseEDL(content);
                    break;
                    
                case 'ale':
                    result = P5Index.util.ALEParser.parseALE(content);
                    break;
                    
                case 'fcpxml':
                    result = P5Index.util.FCPXMLParser.parseFCPXML(content);
                    break;
                    
                case 'xml':
                    result = this.parseXMLContent(content);
                    break;
                    
                case 'otio':
                case 'json':
                    result = P5Index.util.OTIOParser.parseOTIO(content);
                    break;
                    
                case 'tsv':
                    result = P5Index.util.TSVParser.parseTSV(content);
                    break;
                    
                case 'txt':
                case 'log':
                    result = P5Index.util.TEXTParser.parseTEXT(content);
                    break;
                    
                default:
                    // NEU: Log unsupported format
                    if (logger.isLogging) {
                        logger.addLog('Unsupported format: ' + format, 'ERROR');
                    }
                    result = { 
                        success: false, 
                        error: `Unsupported format: ${format}. Supported: EDL, ALE, FCPXML, XML, OTIO, JSON, TXT, TSV, LOG`
                    };
            }
        } catch (parseError) {
            // NEU: Log parse error
            if (logger.isLogging) {
                logger.addLog('Parse exception: ' + parseError.message, 'ERROR');
            }
            console.error(`Error parsing ${format} file:`, parseError);
            result = {
                success: false,
                error: `Parse error: ${parseError.message}`
            };
        }
        
        // CUSTOMER FILTER ANWENDEN
        if (result.success && result.mediaItems && enableArchiveFilter) {
            var originalItems = result.mediaItems.slice(); // Copy für Statistiken
            var originalCount = originalItems.length;
            
            // NEU: Log before filter
            if (logger.isLogging && debugMode) {
                logger.addLog('Applying MediaCustomerFilter to ' + originalCount + ' items');
            }
            
            result.mediaItems = P5Index.util.MediaCustomerFilter.filterForArchive(
                result.mediaItems, 
                result.formatType
            );
            
            var filteredCount = result.mediaItems.length;
            var removedCount = originalCount - filteredCount;
            
            if (removedCount > 0) {
                result.detectedFormat += ' (Archive Filtered)';
                result.filteredItems = removedCount;
                result.totalCount = filteredCount;
                
                // NEU: Log filter result
                if (logger.isLogging) {
                    logger.addLog('MediaCustomerFilter removed ' + removedCount + ' items (' + 
                                Math.round((removedCount/originalCount)*100) + '%)');
                }
                
                console.log(`Customer filter: Removed ${removedCount} non-media items (${Math.round((removedCount/originalCount)*100)}%)`);
                
                // Detaillierte Statistiken (optional)
                if (enableStatistics) {
                    result.filterStatistics = P5Index.util.MediaCustomerFilter.getFilterStatistics(
                        originalItems, result.mediaItems, result.detectedFormat
                    );
                }
            }
        }
        
        // METADATA ANREICHERN
        if (result.success) {
            var parseTime = Date.now() - startTime;
            
            result.sourceFile = fileName;
            result.parsedAt = new Date().toISOString();
            result.parseTimeMs = parseTime;
            
            // sourceFormat zu allen Items hinzufügen
            if (result.mediaItems) {
                result.mediaItems.forEach(function(item) {
                    item.sourceFormat = result.detectedFormat;
                    item.sourceFile = fileName;
                });
            }
            
            // NEU: Log success
            if (logger.isLogging) {
                logger.addLog('OptimizedMediaParser: Success - ' + 
                            (result.totalCount || result.mediaItems.length) + 
                            ' items ready for archive search (' + parseTime + 'ms)');
            }
            
            console.log(`Parsing completed in ${parseTime}ms: ${result.totalCount || result.mediaItems.length} media items`);
        } else {
            // NEU: Log failure
            if (logger.isLogging) {
                logger.addLog('OptimizedMediaParser: Failed - ' + result.error, 'ERROR');
            }
            console.error(`Parsing failed: ${result.error}`);
        }
        
        return result;
    },

    /**
     * XML Content Parser mit Auto-Detection
     */
    parseXMLContent: function(content) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        // XMEML Detection
        if (this.isXMEML(content)) {
            // NEU: Log XMEML detection
            if (logger.isLogging && debugMode) {
                logger.addLog('Detected XMEML content → using XMEMLParser');
            }
            console.log('XMEML format detected, trying XMEML parser...');
            return P5Index.util.XMEMLParser.parseXMEML(content);
         }
        
        // Andere XML-Formate
        if (content.includes('<fcpxml')) {
            // Fallback: versuche FCPXML parser
            // NEU: Log FCPXML fallback
            if (logger.isLogging && debugMode) {
                logger.addLog('XML appears to be FCPXML → using FCPXMLParser');
            }
            console.log('XML file appears to be FCPXML, trying FCPXML parser...');
            return P5Index.util.FCPXMLParser.parseFCPXML(content);
        }
        
        // NEU: Log unknown XML
        if (logger.isLogging) {
            logger.addLog('Unknown XML format', 'WARNING');
        }
        
        return { 
            success: false, 
            error: 'Unsupported XML format. Only FCPXML and XMEML are supported.' 
        };
    },

    /**
     * XMEML Format Detection
     */
    isXMEML: function(xmlContent) {
        return xmlContent.includes('<!DOCTYPE xmeml>') || 
               xmlContent.includes('<xmeml') ||
               (xmlContent.includes('xmlns') && xmlContent.includes('xmeml'));
    },

    // ===== CONVENIENCE METHODS =====

    /**
     * Parse mit Standard-Einstellungen für Archiv-Index
     * (Diese Funktion wird vom Controller verwendet)
     */
    parseForArchiveIndex: function(content, format, fileName) {
        return this.parseForArchive(content, format, fileName, {
            enableArchiveFilter: true,
            enableStatistics: false
        });
    },

    /**
     * Parse mit detaillierten Statistiken für Debugging
     */
    parseWithStatistics: function(content, format, fileName) {
        return this.parseForArchive(content, format, fileName, {
            enableArchiveFilter: true,
            enableStatistics: true
        });
    },

    /**
     * Parse ohne Filter (Original-Verhalten)
     */
    parseWithoutFilter: function(content, format, fileName) {
        return this.parseForArchive(content, format, fileName, {
            enableArchiveFilter: false,
            enableStatistics: false
        });
    },

    // ===== BATCH PROCESSING SUPPORT =====

    /**
     * Batch-Processing für mehrere Dateien
     */
    parseBatch: function(files, options) {
        var logger = P5Index.util.ParserLogger;  // NEU
        
        options = options || {};
        var results = [];
        var startTime = Date.now();
        
        // NEU: Log batch start
        if (logger.isLogging) {
            logger.addLog('Starting batch processing of ' + files.length + ' files');
        }
        
        console.log(`🔍 Starting batch processing of ${files.length} files`);
        
        files.forEach(function(file, index) {
            console.log(`Processing file ${index + 1}/${files.length}: ${file.name}`);
            
            try {
                var format = this.detectFormatFromFileName(file.name);
                var result = this.parseForArchive(file.content, format, file.name, options);
                
                result.batchIndex = index;
                result.batchTotal = files.length;
                results.push(result);
                
            } catch (error) {
                // NEU: Log batch item error
                if (logger.isLogging) {
                    logger.addLog('Batch item ' + (index + 1) + ' failed: ' + error.message, 'ERROR');
                }
                console.error(`Error processing file ${file.name}:`, error);
                results.push({
                    success: false,
                    error: error.message,
                    sourceFile: file.name,
                    batchIndex: index,
                    batchTotal: files.length
                });
            }
        }, this);
        
        var totalTime = Date.now() - startTime;
        var successCount = results.filter(r => r.success).length;
        
        // NEU: Log batch summary
        if (logger.isLogging) {
            logger.addLog('Batch processing completed: ' + successCount + '/' + files.length + 
                        ' successful (' + totalTime + 'ms)');
        }
        
        console.log(`Batch processing completed in ${totalTime}ms: ${successCount}/${files.length} files successful`);
        
        return {
            success: true,
            results: results,
            summary: {
                totalFiles: files.length,
                successfulFiles: successCount,
                failedFiles: files.length - successCount,
                totalProcessingTime: totalTime,
                averageTimePerFile: Math.round(totalTime / files.length)
            }
        };
    },

    /**
     * Format Detection basierend auf Dateiname
     */
    detectFormatFromFileName: function(fileName) {
        var ext = fileName.split('.').pop().toLowerCase();
        
        var formatMap = {
            'edl': 'edl',
            'ale': 'ale',
            'fcpxml': 'fcpxml',
            'xmeml': 'xml',
            'xml': 'xml',
            'otio': 'otio',
            'json': 'json',
            'txt': 'txt',
            'tsv': 'tsv',
            'log': 'log'
        };
        
        return formatMap[ext] || 'unknown';
    },

    // ===== VALIDATION & UTILITIES =====

    /**
     * Validiere Parse-Ergebnis
     */
    validateParseResult: function(result) {
        if (!result) {
            return { valid: false, error: 'No result provided' };
        }
        
        if (!result.success) {
            return { valid: false, error: result.error || 'Parse failed' };
        }
        
        if (!result.mediaItems || !Array.isArray(result.mediaItems)) {
            return { valid: false, error: 'No media items array found' };
        }
        
        if (result.mediaItems.length === 0) {
            return { valid: false, error: 'No media items found in file' };
        }
        
        // Validiere Media Items
        var invalidItems = result.mediaItems.filter(function(item) {
            return !item.fileName || typeof item.fileName !== 'string';
        });
        
        if (invalidItems.length > 0) {
            return { 
                valid: false, 
                error: `${invalidItems.length} media items missing fileName` 
            };
        }
        
        return { 
            valid: true, 
            itemCount: result.mediaItems.length,
            format: result.detectedFormat
        };
    },

    /**
     * Erstelle Summary für UI
     */
    createParseSummary: function(result) {
        if (!result.success) {
            return {
                status: 'error',
                message: result.error,
                details: null
            };
        }
        
        var itemCount = result.mediaItems.length;
        var filteredCount = result.filteredItems || 0;
        var originalCount = itemCount + filteredCount;
        
        return {
            status: 'success',
            message: `Found ${itemCount} media files in ${result.detectedFormat}`,
            details: {
                mediaItems: itemCount,
                originalItems: originalCount,
                filteredItems: filteredCount,
                format: result.detectedFormat,
                sourceFile: result.sourceFile,
                parseTime: result.parseTimeMs
            }
        };
    }
});