Ext.define('P5Index.util.MediaCustomerFilter', {
    singleton: true,
    
    debugMode: false,  // NEU: Als Property für externe Steuerung

    /**
     * Haupt-Filter basierend auf Customer Feedback:
     * "only media and graphics, no dissolves, transitions and adjustment layers"
     */
    filterForArchive: function(mediaItems, formatType) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        var originalCount = mediaItems.length;
        
        // NEU: Log start
        if (logger.isLogging) {
            logger.addLog('MediaCustomerFilter: Processing ' + originalCount + 
                         ' items from ' + formatType);
        }
        
        // Statistiken für Logging
        var stats = {
            system: 0,
            effects: 0,
            transitions: 0,
            technical: 0,
            placeholder: 0,
            unreliable: 0,
            other: 0
        };
        
        var filtered = mediaItems.filter(function(item) {
            var result = this.isArchiveRelevantMedia(item, formatType);
            
            // NEU: Track rejection reasons
            if (!result && debugMode) {
                // Analyse warum gefiltert
                var fileName = (item.fileName || '').toLowerCase();
                var source = (item.source || '').toLowerCase();
                
                if (this.isSystemFile(fileName)) {
                    stats.system++;
                } else if (this.isGeneratedEffect(fileName, source)) {
                    stats.effects++;
                } else if (item.editType && item.editType !== 'C') {
                    stats.transitions++;
                } else if (this.isTechnicalReel(item)) {
                    stats.technical++;
                } else if (fileName.includes('placeholder') || fileName.includes('temp')) {
                    stats.placeholder++;
                } else {
                    stats.other++;
                }
                
                // Log erste paar gefilterte Items
                if (logger.isLogging && debugMode) {
                    var totalFiltered = originalCount - filtered.length;
                    if (totalFiltered <= 5) {
                        logger.addLog('Filtered: "' + item.fileName + '" (' + formatType + ')');
                    }
                }
            }
            
            return result;
        }, this);
        
        var filteredCount = originalCount - filtered.length;
        
        // NEU: Log summary
        if (logger.isLogging && filteredCount > 0) {
            var reasons = [];
            for (var reason in stats) {
                if (stats[reason] > 0) {
                    reasons.push(reason + ': ' + stats[reason]);
                }
            }
            
            logger.addLog('MediaCustomerFilter: Removed ' + filteredCount + 
                         ' items (' + reasons.join(', ') + ')');
        } else if (logger.isLogging) {
            logger.addLog('MediaCustomerFilter: No items filtered');
        }
        
        if (filteredCount > 0) {
            console.log(`Customer filter removed ${filteredCount} effect/transition items from ${formatType}`);
        }
        
        return filtered;
    },

    /**
     * Bestimme ob Item für Archiv-Index relevant ist
     */
    isArchiveRelevantMedia: function(item, formatType) {
        var fileName = (item.fileName || '').toLowerCase();
        var source = (item.source || '').toLowerCase();

        // 1. Universelle Ausschlüsse (alle Formate)
        if (this.isSystemFile(fileName) || this.isGeneratedEffect(fileName, source)) {
            return false;
        }

        // 2. Format-spezifische Customer Fixes
        switch (formatType) {
            case 'EDL':
                return this.filterEDL(item);
            case 'ALE':
                return this.filterALE(item);
            default:
                return true; // FCPXML/OTIO sind bereits sauber
        }
    },

    // === UNIVERSELLE FILTER ===
    
    isSystemFile: function(fileName) {
        return fileName.startsWith('.') || 
               fileName.includes('thumbs.db') ||
               fileName.includes('.ds_store') ||
               fileName.includes('.temp');
    },

    isGeneratedEffect: function(fileName, source) {
        if (this.looksLikeRealMediaFile(fileName)) {
            return false; // Nicht filtern!
        }

        var effectKeywords = [
            'solid color', 'color matte', 'slug', 'black', 'bars', 'tone',
            'title', 'text', 'generator', 'countdown', 'dissolve', 'transition',
            'adjustment', 'fade', 'wipe', 'iris', 'clock', 'push', 'slide'
        ];
        
        return effectKeywords.some(keyword => 
            fileName === keyword || // Exakte Übereinstimmung
            fileName.startsWith(keyword + ' ') // Oder am Anfang
        );
    },
    
    // NEU: Hilfsfunktion für Statistiken
    isTechnicalReel: function(item) {
        var source = item.source;
        var reelName = (item.originalReelName || '').toLowerCase();
        
        if (source === 'reel') {
            var technicalReels = ['ax', 'bx', 'aa', 'v', 'a1', 'a2', 'bl', 'black', 'slug'];
            return technicalReels.includes(reelName);
        }
        return false;
    },

    // === EDL-SPEZIFISCHER FILTER (Customer Problem) ===
    
    filterEDL: function(item) {
        var logger = P5Index.util.ParserLogger;  // NEU
        var debugMode = this.debugMode;         // NEU
        
        // Problem: "picked up dissolves, transitions and adjustment layers"
        
        // 1. Edit Type Filter - NUR Cuts (C), keine Dissolves (D), Wipes (W), Keys (K)
        var editType = item.editType;
        if (editType && editType !== 'C') {
            // NEU: Debug log statt console.log
            if (debugMode && logger.isLogging) {
                logger.addLog('EDL: Filtered ' + editType + ' event (' + 
                            this.getEditTypeName(editType) + ') - reel: ' + item.originalReelName);
            }
            console.log(`EDL: Filtered ${editType} event (${this.getEditTypeName(editType)}) - reel: ${item.originalReelName}`);
            return false;
        }

        // 2. Technische Reels ausschließen
        var source = item.source;
        var reelName = (item.originalReelName || '').toLowerCase();
        
        if (source === 'reel') {
            var technicalReels = ['ax', 'bx', 'aa', 'v', 'a1', 'a2', 'bl', 'black', 'slug'];
            if (technicalReels.includes(reelName)) {
                // NEU: Debug log statt console.log
                if (debugMode && logger.isLogging) {
                    logger.addLog('EDL: Filtered technical reel: ' + reelName);
                }
                console.log(`EDL: Filtered technical reel: ${reelName}`);
                return false;
            }
        }

        // 3. Placeholder/Fake Clips
        var fileName = item.fileName.toLowerCase();
        if (fileName.includes('faux') || fileName.includes('temp') || fileName.includes('placeholder')) {
            // NEU: Debug log statt console.log
            if (debugMode && logger.isLogging) {
                logger.addLog('EDL: Filtered placeholder: ' + fileName);
            }
            console.log(`EDL: Filtered placeholder: ${fileName}`);
            return false;
        }

        return true;
    },

    getEditTypeName: function(editType) {
        var types = {
            'C': 'Cut',
            'D': 'Dissolve', 
            'W': 'Wipe',
            'K': 'Key/Title',
            'B': 'B-mode'
        };
        return types[editType] || editType;
    },

    // === ALE-SPEZIFISCHER FILTER ===
    
    filterALE: function(item) {
        var source = item.source;
        var fileName = item.fileName;
        
        // Fallback Sources nur wenn sie wie echte Media aussehen
        var unreliableSources = ['fallback', 'tape'];
        if (unreliableSources.includes(source)) {
            return this.looksLikeRealMediaFile(fileName);
        }
        
        return true;
    },

    // === MEDIA FILE VALIDATION ===
    
   looksLikeRealMediaFile: function(fileName) {
        if (!fileName || fileName.length < 3) return false;
        
        // 1. Hat Media Extension
        if (fileName.includes('.')) {
            var ext = fileName.split('.').pop().toLowerCase();
            var mediaExtensions = [
                // Video Extensions
                'mov', 'mp4', 'avi', 'mxf', 'r3d', 'braw', 'dng', 'ari', 'cine', 'mkv', 'wmv',
                // Audio Extensions  
                'wav', 'aiff', 'mp3', 'aac', 'flac', 'ogg',
                // Graphics Extensions
                'jpg', 'jpeg', 'png', 'tiff', 'tif', 'psd', 'ai', 'exr', 'dpx', 'tga', 'bmp', 'gif'
            ];
            return mediaExtensions.includes(ext);
        }
        
        // 2. Camera/Media naming patterns (ohne Extension)
        if (fileName.match(/^[A-Za-z]\d+[A-Za-z]\d+/)) return true; // A001C002_take1
        if (fileName.match(/^\d{4}-\d{2}-\d{2}/)) return true; // 2024-01-15_scene
        if (fileName.match(/^[A-Z]{2,4}_\d{3,4}/)) return true; // CAM_001, ARRI_1234
        
        return false;
    },
    
    // NEU: Statistik-Methode (falls gewünscht)
    getFilterStatistics: function(originalItems, filteredItems, format) {
        var removed = originalItems.filter(function(item) {
            return !filteredItems.includes(item);
        });
        
        var statistics = {
            format: format,
            original: originalItems.length,
            filtered: filteredItems.length,
            removed: removed.length,
            removedTypes: {}
        };
        
        // Kategorisiere entfernte Items
        removed.forEach(function(item) {
            var type = this.categorizeRemovedItem(item);
            statistics.removedTypes[type] = (statistics.removedTypes[type] || 0) + 1;
        }, this);
        
        return statistics;
    },
    
    // NEU: Hilfsmethode für Statistiken
    categorizeRemovedItem: function(item) {
        var fileName = (item.fileName || '').toLowerCase();
        var source = (item.source || '').toLowerCase();
        
        if (this.isSystemFile(fileName)) return 'system_files';
        if (this.isGeneratedEffect(fileName, source)) return 'effects_transitions';
        if (item.editType && item.editType !== 'C') return 'non_cut_edits';
        if (this.isTechnicalReel(item)) return 'technical_reels';
        if (fileName.includes('placeholder') || fileName.includes('temp')) return 'placeholders';
        
        return 'other';
    }
});