Ext.define('P5Index.util.ALEParser', {
    singleton: true,
    
    debugMode: false,  // Als Property für externe Steuerung

    /**
     * Parse ALE (Avid Log Exchange) file and extract media references
     * @param {string} aleContent - The ALE content as string
     * @returns {Object} Object containing extracted media items
     */
    parseALE: function(aleContent) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        try {
            var lines = aleContent.split(/\r\n|\r|\n/);
            var mediaItems = [];
            var uniqueFiles = {};
            var section = 'heading';
            var columnHeaders = [];
            var columnMapping = {};
            var metadata = {};
            var dataLineCount = 0;
            var skippedCount = 0;
            
            // Log parse start
            if (logger.isLogging) {
                logger.addLog('Starting ALE parse (' + lines.length + ' lines)');
            }
            
            console.log('Parsing ALE with', lines.length, 'lines');
            
            for (var i = 0; i < lines.length; i++) {
                var line = lines[i];
                
                // Nur für Section-Detection trimmen
                var trimmedLine = line.trim();
                
                if (trimmedLine === 'Heading') {
                    section = 'heading';
                    continue;
                } else if (trimmedLine === 'Column') {
                    section = 'column';
                    // Log metadata when transitioning to column section
                    if (logger.isLogging && Object.keys(metadata).length > 0) {
                        var fps = metadata.FPS || metadata.Fps || 'unknown';
                        var delim = metadata.FIELD_DELIM || 'TABS';
                        logger.addLog('ALE metadata: FPS=' + fps + ', FIELD_DELIM=' + delim);
                    }
                    continue;
                } else if (trimmedLine === 'Data') {
                    section = 'data';
                    // Log column mapping when starting data section
                    if (logger.isLogging) {
                        var mappedCols = [];
                        if (columnMapping.sourceFile !== undefined) mappedCols.push('sourceFile');
                        if (columnMapping.name !== undefined) mappedCols.push('name');
                        if (columnMapping.tape !== undefined) mappedCols.push('tape');
                        if (columnMapping.clip !== undefined) mappedCols.push('clip');
                        if (columnMapping.duration !== undefined) mappedCols.push('duration');
                        
                        if (mappedCols.length > 0) {
                            logger.addLog('Found columns: ' + mappedCols.join(', '));
                        } else {
                            logger.addLog('No standard columns found in ALE', 'WARNING');
                        }
                    }
                    continue;
                }
                
                // Bei Data-Processing die Original-Zeile verwenden
                if (section === 'data') {
                    dataLineCount++;
                    this.parseDataLine(line, columnHeaders, columnMapping, uniqueFiles, metadata, dataLineCount);
                } else {
                    this.parseHeadingLine(trimmedLine, metadata);  // Trimmed für Heading/Column
                    this.parseColumnLine(trimmedLine, columnHeaders, columnMapping);
                }
            }
            
            // Convert unique files to media items array
            mediaItems = Object.values(uniqueFiles);
            
            // Count skipped entries
            skippedCount = dataLineCount - mediaItems.length;
            
            // Log summary
            if (logger.isLogging) {
                if (dataLineCount > 0) {
                    logger.addLog('Processed ' + dataLineCount + ' data lines');
                }
                
                if (mediaItems.length > 0) {
                    // Source type summary
                    var sourceCounts = {};
                    mediaItems.forEach(function(item) {
                        sourceCounts[item.source] = (sourceCounts[item.source] || 0) + 1;
                    });
                    var summary = Object.keys(sourceCounts).map(function(source) {
                        return source + ': ' + sourceCounts[source];
                    }).join(', ');
                    logger.addLog('Found ' + mediaItems.length + ' unique files (' + summary + ')');
                    
                    if (skippedCount > 0) {
                        logger.addLog('Skipped ' + skippedCount + ' entries without valid filenames');
                    }
                }
            }
            
            console.log('ALE parsing completed:', mediaItems.length, 'unique media items found');
            console.log('ALE metadata:', metadata);
            
            return {
                success: true,
                mediaItems: mediaItems,
                totalCount: mediaItems.length,
                formatType: 'ALE',
                detectedFormat: 'Avid Log Exchange (ALE)',
                metadata: metadata
            };
            
        } catch (error) {
            // Log error
            if (logger.isLogging) {
                logger.addLog('ALE parse failed - ' + error.message, 'ERROR');
            }
            console.error('Error parsing ALE content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    /**
     * Parse heading section line
     */
    parseHeadingLine: function(line, metadata) {
        // Handle format like "FIELD_DELIM	TABS" or "FPS	25"
        var parts = line.split('\t');
        if (parts.length >= 2) {
            var key = parts[0].trim();
            var value = parts.slice(1).join('\t').trim();
            metadata[key] = value;
        }
    },

    /**
     * Parse column definition line
     */
    parseColumnLine: function(line, columnHeaders, columnMapping) {
        // Split by tabs to get column headers
        var headers = line.split('\t');
        
        // Store headers and create mapping for key columns
        columnHeaders.length = 0; // Clear array
        for (var i = 0; i < headers.length; i++) {
            var header = headers[i].trim();
            columnHeaders.push(header);
            
            // Map important columns by index
            var headerLower = header.toLowerCase();
            if (headerLower === 'source file' || headerLower === 'sourcefile') {
                columnMapping.sourceFile = i;
            } else if (headerLower === 'name') {
                columnMapping.name = i;
            } else if (headerLower === 'duration') {
                columnMapping.duration = i;
            } else if (headerLower === 'start') {
                columnMapping.start = i;
            } else if (headerLower === 'end') {
                columnMapping.end = i;
            } else if (headerLower === 'tape' || headerLower === 'reel_name') {
                columnMapping.tape = i;
            } else if (headerLower === 'clip') {
                columnMapping.clip = i;
            }
        }
        
        console.log('ALE column mapping:', columnMapping);
    },

    /**
     * Parse data line
     */
    parseDataLine: function(line, columnHeaders, columnMapping, uniqueFiles, metadata, lineNumber) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var values = line.split('\t');
        
        // Skip if not enough values
        if (values.length < columnHeaders.length / 2) {
            return;
        }
        
        // Extract key values using column mapping
        var sourceFile = this.getColumnValue(values, columnMapping.sourceFile);
        var name = this.getColumnValue(values, columnMapping.name);
        var duration = this.getColumnValue(values, columnMapping.duration);
        var start = this.getColumnValue(values, columnMapping.start);
        var end = this.getColumnValue(values, columnMapping.end);
        var tape = this.getColumnValue(values, columnMapping.tape);
        var clip = this.getColumnValue(values, columnMapping.clip);
        
        // Determine best filename to use
        var fileName = this.determineBestFileName(sourceFile, name, clip, tape);
        
        if (!fileName) {
            if (logger.isLogging && debugMode) {
                var nameInfo = name ? '"' + name + '"' : 'no name';
                var tapeInfo = tape ? '"' + tape + '"' : 'no tape';
                logger.addLog('Line ' + lineNumber + ': Skipped - no valid filename (' + 
                            nameInfo + ', ' + tapeInfo + ')', 'WARNING');
            }
            console.log('Skipping ALE entry - no usable filename found');
            return;
        }
        
        var key = fileName.toLowerCase();
        
        if (!uniqueFiles[key]) {
            var source = this.determineSource(sourceFile, name, clip, tape);
            
            if (logger.isLogging && debugMode) {
                logger.addLog('Line ' + lineNumber + ': Found "' + fileName + '" (source: ' + source + ')');
            }
            
            uniqueFiles[key] = {
                fileName: fileName,
                path: this.generatePath(sourceFile, fileName),
                usedDurationTC: this.normalizeDuration(duration),
                availableDurationTC: '', // Unknown from ALE
                source: source,
                originalSourceFile: sourceFile,
                originalName: name,
                clips: []
            };
        }
        
        // Add clip information
        uniqueFiles[key].clips.push({
            name: name,
            clip: clip,
            duration: duration,
            start: start,
            end: end,
            tape: tape,
            rawData: this.createRawDataObject(columnHeaders, values)
        });
    },

    /**
     * Get column value by index
     */
    getColumnValue: function(values, columnIndex) {
        if (columnIndex !== undefined && columnIndex < values.length) {
            var value = values[columnIndex].trim();
            return value || null;
        }
        return null;
    },

    /**
     * Determine best filename from available options
     */
    determineBestFileName: function(sourceFile, name, clip, tape) {
        // Priority 1: Source File (if it looks like a filename)
        if (sourceFile && this.isValidFileName(sourceFile)) {
            return this.getFileName(sourceFile);
        }
        
        // Priority 2: Name (if it looks like a filename)
        if (name && this.isValidFileName(name)) {
            return name;
        }
        
        // Priority 3: Clip (if it looks like a filename)
        if (clip && this.isValidFileName(clip)) {
            return clip;
        }
        
        // Priority 4: Tape/Reel (if it looks like a filename)
        if (tape && this.isValidFileName(tape)) {
            return tape;
        }
        
        // Fallback: Use name even if it doesn't look like filename
        if (name) {
            return name;
        }
        
        return null;
    },

    /**
     * Check if string looks like a valid filename
     */
    isValidFileName: function(str) {
        if (!str || str.length < 3) return false;
        
        // Has file extension
        if (str.includes('.')) return true;
        
        // Looks like camera/media naming convention
        if (str.match(/^[A-Za-z]\d+[A-Za-z]\d+/)) return true;  // A057C001_210929_R4AF
        
        // Has reasonable structure
        if (str.match(/^[A-Za-z0-9][A-Za-z0-9_-]*[A-Za-z0-9]$/)) return true;
        
        return false;
    },

    /**
     * Determine source type
     */
    determineSource: function(sourceFile, name, clip, tape) {
        if (sourceFile && this.isValidFileName(sourceFile)) {
            return 'source_file';
        } else if (name && this.isValidFileName(name)) {
            return 'name';
        } else if (clip && this.isValidFileName(clip)) {
            return 'clip';
        } else if (tape && this.isValidFileName(tape)) {
            return 'tape';
        }
        return 'fallback';
    },

    /**
     * Generate path from source file and filename
     */
    generatePath: function(sourceFile, fileName) {
        // If source file contains path, use it
        if (sourceFile && (sourceFile.includes('/') || sourceFile.includes('\\'))) {
            return sourceFile;
        }
        
        // If source file is different from filename, construct path
        if (sourceFile && sourceFile !== fileName) {
            return sourceFile;
        }
        
        // Default: just filename
        return fileName;
    },

    /**
     * Normalize duration format
     */
    normalizeDuration: function(duration) {
        if (!duration) return '';
        
        // Already in timecode format (HH:MM:SS:FF)
        if (duration.match(/^\d{2}:\d{2}:\d{2}:\d{2}$/)) {
            return duration;
        }
        
        // Handle other formats if needed
        return duration;
    },

    /**
     * Create raw data object for debugging
     */
    createRawDataObject: function(headers, values) {
        var rawData = {};
        for (var i = 0; i < Math.min(headers.length, values.length); i++) {
            if (headers[i] && values[i]) {
                rawData[headers[i]] = values[i].trim();
            }
        }
        return rawData;
    },

    /**
     * Extract filename from path
     * @param {string} path - File path
     * @returns {string} Filename
     */
    getFileName: function(path) {
        if (!path) return '';
        return path.split('/').pop().split('\\').pop();
    }
});