Ext.define('P5Index.util.XMEMLParser', {
    singleton: true,
    
    debugMode: false,  // Als Property für externe Steuerung

    /**
     * Parse XMEML file and extract media references
     * @param {string} xmemlContent - The XMEML content as string
     * @returns {Object} Object containing extracted media items
     */
    parseXMEML: function(xmemlContent) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        try {
            // Log start
            if (logger.isLogging) {
                logger.addLog('Starting XMEML parse');
            }
            
            console.log('Parsing XMEML...');
            
            // Parse XML
            var parser = new DOMParser();
            var xmlDoc = parser.parseFromString(xmemlContent, 'text/xml');
            
            // Check for XML parsing errors
            var parseError = xmlDoc.getElementsByTagName('parsererror');
            if (parseError.length > 0) {
                if (logger.isLogging) {
                    logger.addLog('XMEML XML parsing failed', 'ERROR');
                }
                throw new Error('XML parsing failed: ' + parseError[0].textContent);
            }
            
            var mediaItems = [];
            var uniqueFiles = {};
            var fileRegistry = {}; // Map file IDs to file data
            var sequenceInfo = {};
            
            // Get XMEML version
            var xmemlElement = xmlDoc.getElementsByTagName('xmeml')[0];
            var version = xmemlElement ? xmemlElement.getAttribute('version') : 'unknown';
            
            if (logger.isLogging) {
                logger.addLog('XMEML version: ' + version);
            }
            
            console.log('XMEML version:', version);
            
            // Step 1: Parse sequence info for default framerate
            sequenceInfo = this.parseSequenceInfo(xmlDoc);
            
            if (logger.isLogging && sequenceInfo.name) {
                logger.addLog('Sequence: "' + sequenceInfo.name + '", FPS=' + sequenceInfo.defaultFps);
            }
            
            // Step 2: Parse all file elements
            this.parseFiles(xmlDoc, fileRegistry, sequenceInfo.defaultFps);
            console.log('Found', Object.keys(fileRegistry).length, 'files in XMEML');
            
            // Step 3: Get clip counts
            var clipCount = xmlDoc.getElementsByTagName('clipitem').length;
            var generatorCount = xmlDoc.getElementsByTagName('generatoritem').length;
            
            if (logger.isLogging) {
                logger.addLog('Structure: ' + Object.keys(fileRegistry).length + ' files, ' + 
                             clipCount + ' clips, ' + generatorCount + ' generators');
            }
            
            // Step 4: Parse clip usage from timeline
            this.parseClipUsage(xmlDoc, fileRegistry, uniqueFiles, sequenceInfo.defaultFps);
            
            // Step 5: Add unused files (available but not used in timeline)
            this.addUnusedFiles(fileRegistry, uniqueFiles);
            
            // Convert to array
            mediaItems = Object.values(uniqueFiles);
            
            // Log usage summary
            if (logger.isLogging) {
                var usedCount = 0;
                var unusedCount = 0;
                var generatedCount = 0;
                
                mediaItems.forEach(function(item) {
                    if (item.source === 'unused_file') {
                        unusedCount++;
                    } else if (item.source === 'generatoritem') {
                        generatedCount++;
                    } else {
                        usedCount++;
                    }
                });
                
                var summary = [];
                if (usedCount > 0) summary.push('used: ' + usedCount);
                if (unusedCount > 0) summary.push('unused: ' + unusedCount);
                if (generatedCount > 0) summary.push('generated: ' + generatedCount);
                
                logger.addLog('Found ' + mediaItems.length + ' items (' + summary.join(', ') + ')');
            }
            
            console.log('XMEML parsing completed:', mediaItems.length, 'unique media items found');
            
            return {
                success: true,
                mediaItems: mediaItems,
                totalCount: mediaItems.length,
                formatType: 'XMEML',
                detectedFormat: 'Final Cut Pro XML (XMEML v' + version + ')',
                metadata: {
                    version: version,
                    fileCount: Object.keys(fileRegistry).length,
                    sequenceName: sequenceInfo.name,
                    defaultFps: sequenceInfo.defaultFps
                }
            };
            
        } catch (error) {
            if (logger.isLogging) {
                logger.addLog('XMEML parse failed - ' + error.message, 'ERROR');
            }
            console.error('Error parsing XMEML content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    /**
     * Parse sequence information for default settings
     */
    parseSequenceInfo: function(xmlDoc) {
        var sequences = xmlDoc.getElementsByTagName('sequence');
        var defaultFps = 25; // Default fallback
        var sequenceName = '';
        
        if (sequences.length > 0) {
            var sequence = sequences[0];
            
            // Get sequence name
            var nameElement = sequence.getElementsByTagName('name')[0];
            if (nameElement) {
                sequenceName = nameElement.textContent;
            }
            
            // Get framerate from rate element
            var rateElement = sequence.getElementsByTagName('rate')[0];
            if (rateElement) {
                var timebaseElement = rateElement.getElementsByTagName('timebase')[0];
                if (timebaseElement) {
                    defaultFps = parseInt(timebaseElement.textContent) || 25;
                }
            }
        }
        
        return {
            name: sequenceName,
            defaultFps: defaultFps
        };
    },

    /**
     * Parse all file elements from XMEML
     */
    parseFiles: function(xmlDoc, fileRegistry, defaultFps) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var files = xmlDoc.getElementsByTagName('file');
        
        for (var i = 0; i < files.length; i++) {
            var file = files[i];
            var fileData = this.extractFileData(file, defaultFps);
            
            if (fileData && fileData.id) {
                fileRegistry[fileData.id] = fileData;
                
                if (logger.isLogging && debugMode) {
                    logger.addLog('File ID ' + fileData.id + ': "' + fileData.fileName + '"' + 
                                (fileData.hasVideo ? ' [video]' : '') + 
                                (fileData.hasAudio ? ' [audio]' : ''));
                }
                
                console.log('Added file:', fileData.fileName, '(ID:', fileData.id + ')');
            }
        }
    },

    /**
     * Extract data from file element
     */
    extractFileData: function(fileElement, defaultFps) {
        var id = fileElement.getAttribute('id');
        var nameElement = fileElement.getElementsByTagName('name')[0];
        var pathurlElement = fileElement.getElementsByTagName('pathurl')[0];
        var durationElement = fileElement.getElementsByTagName('duration')[0];
        
        if (!id) {
            console.log('Skipping file without ID');
            return null;
        }
        
        var fileName = '';
        var filePath = '';
        var duration = '';
        var fps = defaultFps;
        
        // Extract filename
        if (nameElement) {
            fileName = nameElement.textContent;
        }
        
        // Extract and decode file path
        if (pathurlElement) {
            var rawPath = pathurlElement.textContent;
            filePath = this.decodeFileUrl(rawPath);
            
            // If no filename from name element, extract from path
            if (!fileName) {
                fileName = this.extractFileName(filePath);
            }
        }
        
        // Get duration in frames
        if (durationElement) {
            duration = durationElement.textContent;
        }
        
        // Try to get FPS from file's rate element
        var rateElement = fileElement.getElementsByTagName('rate')[0];
        if (rateElement) {
            var timebaseElement = rateElement.getElementsByTagName('timebase')[0];
            if (timebaseElement) {
                fps = parseInt(timebaseElement.textContent) || defaultFps;
            }
        }
        
        // Get media type information
        var hasVideo = false;
        var hasAudio = false;
        var width = '';
        var height = '';
        
        var mediaElement = fileElement.getElementsByTagName('media')[0];
        if (mediaElement) {
            var videoElement = mediaElement.getElementsByTagName('video')[0];
            if (videoElement) {
                hasVideo = true;
                var sampleChar = videoElement.getElementsByTagName('samplecharacteristics')[0];
                if (sampleChar) {
                    var widthEl = sampleChar.getElementsByTagName('width')[0];
                    var heightEl = sampleChar.getElementsByTagName('height')[0];
                    if (widthEl) width = widthEl.textContent;
                    if (heightEl) height = heightEl.textContent;
                }
            }
            
            var audioElement = mediaElement.getElementsByTagName('audio')[0];
            if (audioElement) {
                hasAudio = true;
            }
        }
        
        if (!fileName) {
            console.log('Skipping file element without usable filename');
            return null;
        }
        
        return {
            id: id,
            fileName: fileName,
            filePath: filePath,
            duration: duration,
            availableDurationTC: this.framesToTimecode(duration, fps),
            fps: fps,
            hasVideo: hasVideo,
            hasAudio: hasAudio,
            width: width,
            height: height,
            used: false // Will be set to true if used in timeline
        };
    },

    /**
     * Parse clip usage from timeline
     */
    parseClipUsage: function(xmlDoc, fileRegistry, uniqueFiles, defaultFps) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Parse clipitem elements (video clips)
        var clipItems = xmlDoc.getElementsByTagName('clipitem');
        var processedClips = 0;
        var skippedClips = 0;
        
        for (var i = 0; i < clipItems.length; i++) {
            var result = this.processClipItem(clipItems[i], fileRegistry, uniqueFiles, defaultFps);
            if (result) {
                processedClips++;
            } else {
                skippedClips++;
            }
        }
        
        if (logger.isLogging && debugMode && skippedClips > 0) {
            logger.addLog('Processed ' + processedClips + ' clips, skipped ' + skippedClips);
        }
        
        // Parse generatoritem elements (solid colors, titles, etc.)
        var generatorItems = xmlDoc.getElementsByTagName('generatoritem');
        for (var j = 0; j < generatorItems.length; j++) {
            this.processGeneratorItem(generatorItems[j], uniqueFiles, defaultFps);
        }
    },

    /**
     * Process individual clipitem element
     */
    processClipItem: function(clipElement, fileRegistry, uniqueFiles, defaultFps) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var name = '';
        var start = '';
        var end = '';
        var duration = '';
        var inPoint = '';
        var outPoint = '';
        var fileRef = null;
        var fps = defaultFps;
        
        // Extract clip properties
        var nameElement = clipElement.getElementsByTagName('name')[0];
        if (nameElement) {
            name = nameElement.textContent;
        }
        
        var startAttr = clipElement.getAttribute('start');
        var endAttr = clipElement.getAttribute('end');
        var durationElement = clipElement.getElementsByTagName('duration')[0];
        var inElement = clipElement.getElementsByTagName('in')[0];
        var outElement = clipElement.getElementsByTagName('out')[0];
        
        if (startAttr) start = startAttr;
        if (endAttr) end = endAttr;
        if (durationElement) duration = durationElement.textContent;
        if (inElement) inPoint = inElement.textContent;
        if (outElement) outPoint = outElement.textContent;
        
        // Find referenced file
        var fileElement = clipElement.getElementsByTagName('file')[0];
        if (fileElement) {
            var fileId = fileElement.getAttribute('id');
            if (fileId && fileRegistry[fileId]) {
                fileRef = fileRegistry[fileId];
                fileRef.used = true;
                fps = fileRef.fps || defaultFps;
            } else if (logger.isLogging && debugMode) {
                logger.addLog('Clip "' + name + '": Unknown file ID ' + fileId, 'WARNING');
            }
        }
        
        var fileName = '';
        var filePath = '';
        
        if (fileRef) {
            fileName = fileRef.fileName;
            filePath = fileRef.filePath;
        } else if (name) {
            // Fallback to clip name if no file reference
            fileName = name;
            if (logger.isLogging && debugMode) {
                logger.addLog('Clip "' + name + '": Using clip name as filename (no file ref)', 'WARNING');
            }
        }
        
        if (!fileName) {
            return false;
        }
        
        if (logger.isLogging && debugMode) {
            logger.addLog('Processing clip: "' + fileName + '" (duration: ' + duration + ' frames)');
        }
        
        var key = fileName.toLowerCase();
        
        if (!uniqueFiles[key]) {
            uniqueFiles[key] = {
                fileName: fileName,
                path: filePath,
                usedDurationTC: this.framesToTimecode(duration, fps),
                availableDurationTC: fileRef ? fileRef.availableDurationTC : '',
                source: 'clipitem',
                mediaKind: 'original-media',
                clips: [],
                hasVideo: fileRef ? fileRef.hasVideo : true,
                hasAudio: fileRef ? fileRef.hasAudio : false,
                width: fileRef ? fileRef.width : '',
                height: fileRef ? fileRef.height : ''
            };
        }
        
        // Add clip usage information
        uniqueFiles[key].clips.push({
            elementType: 'clipitem',
            name: name,
            start: start,
            end: end,
            duration: duration,
            inPoint: inPoint,
            outPoint: outPoint,
            startTC: this.framesToTimecode(start, fps),
            durationTC: this.framesToTimecode(duration, fps),
            fileId: fileRef ? fileRef.id : null
        });
        
        // Update used duration to longest clip if multiple
        var currentDurationTC = this.framesToTimecode(duration, fps);
        if (currentDurationTC && (!uniqueFiles[key].usedDurationTC || 
            this.compareTimecodes(currentDurationTC, uniqueFiles[key].usedDurationTC, fps) > 0)) {
            uniqueFiles[key].usedDurationTC = currentDurationTC;
        }
        
        return true;
    },

    /**
     * Process generator items (titles, solid colors, etc.)
     */
    processGeneratorItem: function(genElement, uniqueFiles, defaultFps) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var name = '';
        var duration = '';
        var fps = defaultFps;
        
        var nameElement = genElement.getElementsByTagName('name')[0];
        if (nameElement) {
            name = nameElement.textContent;
        }
        
        var durationElement = genElement.getElementsByTagName('duration')[0];
        if (durationElement) {
            duration = durationElement.textContent;
        }
        
        // Get FPS from rate if available
        var rateElement = genElement.getElementsByTagName('rate')[0];
        if (rateElement) {
            var timebaseElement = rateElement.getElementsByTagName('timebase')[0];
            if (timebaseElement) {
                fps = parseInt(timebaseElement.textContent) || defaultFps;
            }
        }
        
        if (name) {
            if (logger.isLogging && debugMode) {
                logger.addLog('Processing generator: "' + name + '"');
            }
            
            var key = name.toLowerCase();
            
            if (!uniqueFiles[key]) {
                uniqueFiles[key] = {
                    fileName: name,
                    path: '', // Generators don't have file paths
                    usedDurationTC: this.framesToTimecode(duration, fps),
                    availableDurationTC: '',
                    source: 'generatoritem',
                    mediaKind: 'generated',
                    clips: [],
                    hasVideo: true,
                    hasAudio: false
                };
            }
        }
    },

    /**
     * Add unused files to the file list
     */
    addUnusedFiles: function(fileRegistry, uniqueFiles) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        var unusedCount = 0;
        
        for (var fileId in fileRegistry) {
            var file = fileRegistry[fileId];
            
            if (!file.used && file.fileName) {
                unusedCount++;
                var key = file.fileName.toLowerCase();
                
                if (!uniqueFiles[key]) {
                    if (logger.isLogging && debugMode) {
                        logger.addLog('Adding unused file: "' + file.fileName + '"');
                    }
                    
                    uniqueFiles[key] = {
                        fileName: file.fileName,
                        path: file.filePath,
                        usedDurationTC: '', // Not used in timeline
                        availableDurationTC: file.availableDurationTC,
                        source: 'unused_file',
                        mediaKind: 'original-media',
                        clips: [],
                        hasVideo: file.hasVideo,
                        hasAudio: file.hasAudio,
                        width: file.width,
                        height: file.height
                    };
                }
            }
        }
        
        if (logger.isLogging && debugMode && unusedCount > 0) {
            logger.addLog('Added ' + unusedCount + ' unused files');
        }
    },

    /**
     * Convert frames to timecode (HH:MM:SS:FF)
     */
    framesToTimecode: function(frames, fps) {
        if (!frames || frames === '' || isNaN(frames)) return '';
        
        fps = fps || 25; // Default fallback
        var totalFrames = parseInt(frames);
        
        var hours = Math.floor(totalFrames / (fps * 3600));
        var minutes = Math.floor((totalFrames % (fps * 3600)) / (fps * 60));
        var seconds = Math.floor((totalFrames % (fps * 60)) / fps);
        var frameRemainder = totalFrames % fps;
        
        return this.padZero(hours) + ':' + 
               this.padZero(minutes) + ':' + 
               this.padZero(seconds) + ':' + 
               this.padZero(frameRemainder);
    },

    /**
     * Compare two timecode strings
     */
    compareTimecodes: function(tc1, tc2, fps) {
        // Convert to frames for comparison
        var frames1 = this.timecodeToFrames(tc1, fps);
        var frames2 = this.timecodeToFrames(tc2, fps);
        
        return frames1 - frames2;
    },

    /**
     * Convert timecode to frames
     */
    timecodeToFrames: function(timecode, fps) {
        if (!timecode) return 0;
        
        fps = fps || 25; // Default
        
        var parts = timecode.split(':');
        if (parts.length === 4) {
            var hours = parseInt(parts[0]) || 0;
            var minutes = parseInt(parts[1]) || 0;
            var seconds = parseInt(parts[2]) || 0;
            var frames = parseInt(parts[3]) || 0;
            
            return (hours * 3600 + minutes * 60 + seconds) * fps + frames;
        }
        
        return 0;
    },

    /**
     * Decode file:// URL and extract clean path
     */
    decodeFileUrl: function(fileUrl) {
        if (!fileUrl) return '';
        
        try {
            // Handle different URL schemes
            if (fileUrl.startsWith('file://')) {
                // File URL: file:///Volumes/Media/file.mov
                var path = fileUrl.replace(/^file:\/\//, '');
                // Keep leading slash for absolute paths on Unix/Mac
                if (!path.startsWith('/') && !path.match(/^[A-Za-z]:/)) {
                    path = '/' + path;
                }
                return decodeURIComponent(path);
            } else {
                // Assume it's already a clean path or relative path
                return decodeURIComponent(fileUrl);
            }
        } catch (e) {
            console.warn('Could not decode file URL:', fileUrl);
            return fileUrl;
        }
    },

    /**
     * Extract filename from path
     */
    extractFileName: function(path) {
        if (!path) return '';
        
        // Handle both forward and backward slashes
        var fileName = path.split('/').pop().split('\\').pop();
        return fileName;
    },

    /**
     * Pad number with leading zero
     */
    padZero: function(num) {
        return (num < 10 ? '0' : '') + num;
    },

    /**
     * Detect XML format type (XMEML vs other XML formats)
     * @param {string} xmlContent - The XML content to analyze
     * @returns {string} 'xmeml', 'other', or 'invalid'
     */
    detectXMLFormat: function(xmlContent) {
        try {
            // Quick check for XMEML indicators without full parsing
            if (xmlContent.includes('<!DOCTYPE xmeml>') || 
                xmlContent.includes('<xmeml') ||
                xmlContent.includes('xmlns:xsi') && xmlContent.includes('xmeml')) {
                return 'xmeml';
            }
            
            return 'other';
        } catch (e) {
            return 'invalid';
        }
    },

    /**
     * Process an XMEML file and create an ExtJS store from it
     * @param {File} file - The XMEML file from input or drop event
     * @returns {Promise<Ext.data.Store>} Promise resolving to the media store
     */
    processXMEMLFile: function(file) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            
            reader.onload = (event) => {
                try {
                    const content = event.target.result;
                    const result = this.parseXMEML(content);
                    
                    if (!result.success) {
                        reject(new Error(result.error));
                        return;
                    }
                    
                    // Create and return a store with the media items
                    const mediaStore = Ext.create('Ext.data.Store', {
                        fields: [
                            {name: 'fileName', type: 'string'},
                            {name: 'path', type: 'string'},
                            {name: 'usedDurationTC', type: 'string'},
                            {name: 'availableDurationTC', type: 'string'},
                            {name: 'source', type: 'string'},
                            {name: 'mediaKind', type: 'string'},
                            {name: 'hasVideo', type: 'boolean'},
                            {name: 'hasAudio', type: 'boolean'},
                            {name: 'width', type: 'string'},
                            {name: 'height', type: 'string'},
                            {name: 'clips', type: 'auto'}
                        ],
                        data: result.mediaItems
                    });
                    
                    resolve(mediaStore);
                } catch (error) {
                    reject(error);
                }
            };
            
            reader.onerror = (error) => {
                reject(error);
            };
            
            reader.readAsText(file);
        });
    }
});