Ext.define('P5Index.util.EDLParser', {
    singleton: true,
    
    debugMode: false,
    
    // Camera Pattern Definitionen
    CAMERA_PATTERNS: [
        // ARRI / Canon / BMD (klassisch + flexible Suffixe)
        // Matches: A001C001, A001C001_20220216, A006C025_2202216G_48FPS, B001C023_v2_GRADE
        // Extension optional: A001C001.MXF auch über andere Regel
        { 
            vendor: 'ARRI/Canon/BMD', 
            rx: /^[A-Z]\d{3,}C\d{3,}(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 90 
        },
        
        // RED (A-Prefix Format mit flexiblen Suffixen)
        // Matches: A_001C001, A_001C001_20220216_R3D2, A_123C456_240125_REDCODE
        { 
            vendor: 'RED', 
            rx: /^[A-Z]_\d{3,}C\d{3,}(?:_[A-Z0-9]{4,16})*$/i,
            confidence: 90 
        },
        
        // RED (Alternative Format)
        // Matches: A001_C001, A001_C001_20220216_R3D, R123_C456_240125_HQ
        { 
            vendor: 'RED', 
            rx: /^[A-Z]\d{3,}_C\d{3,}(?:_[A-Z0-9]{3,16})*$/i,
            confidence: 85 
        },
        
        // Blackmagic BRAW Extended (mit Zeit + flexiblen Suffixen)
        // Matches: B_0124C001_241112_123439_H1CRO, A_0001C001_250101_093045_BRAW
        { 
            vendor: 'Blackmagic BRAW Extended', 
            rx: /^[A-Z]_\d{3,}C\d{3,}_\d{6}_\d{6}(?:_[A-Z0-9]{4,16})*$/i,
            confidence: 85 
        },
        
        // Blackmagic BRAW Standard (mit flexiblen Suffixen)
        // Matches: B_001C001, B_001C001_20220216_BRAW, B_0124C001_241112_4K
        { 
            vendor: 'Blackmagic BRAW', 
            rx: /^B_\d{3,}C\d{3,}(?:_[A-Z0-9]{4,16})*$/i,
            confidence: 85 
        },
        
        // Sony Venice/FS7/CineAlta (mit flexiblen Suffixen)
        // Matches: C0001A001, C0001A001_20220216_S001, C123B456_240125_XAVC_4K
        { 
            vendor: 'Sony', 
            rx: /^C\d{3,5}[A-Z]?\d{3,}(?:_[A-Z0-9]{3,16})*$/i,
            confidence: 80 
        },
        
        // Sony (Kürzeres Format mit optionalen Suffixen)
        // Matches: C12345, CA12345, CA12345_S001, V123456_LOG
        { 
            vendor: 'Sony', 
            rx: /^(?:C|V)[A-Z]?\d{5,8}(?:_[A-Z0-9]{3,16})*$/i,
            confidence: 75 
        },
        
        // GoPro (mit Extension - spezifisch)
        // Matches: GOPR0001.MP4, GP010123.MOV, GOPR1234.mp4
        { 
            vendor: 'GoPro', 
            rx: /^(?:GOPR\d{4,}|GP\d{2}\d{4,})\.(?:MP4|MOV)$/i,
            confidence: 85 
        },
        
        // DJI (mit Extension - spezifisch)
        // Matches: DJI_0001.MP4, DJI_1234.MOV
        { 
            vendor: 'DJI', 
            rx: /^DJI_\d{4,}\.(?:MP4|MOV)$/i,
            confidence: 85 
        },
        
        // Panasonic VariCam/P2 (mit optionalen Suffixen)
        // Matches: A001L1, MXF01L2, MXF01L2_001, CLIP01L1_24P
        { 
            vendor: 'Panasonic', 
            rx: /^[A-Z0-9]{3,6}L\d(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 70 
        },
        
        // Panasonic (Alternative Format)
        // Matches: 001ABS, 123XYS, 123XYS_001
        { 
            vendor: 'Panasonic', 
            rx: /^\d{3}[A-Z0-9]{2}S(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 65 
        },
        
        // Blackmagic Cintel Scanner (Datum-basiert mit Beschreibung)
        // Matches: Cintel_2024-01-15_Reel01, Cintel_2025-03-20_Film-ABC_4K
        { 
            vendor: 'BMD Cintel', 
            rx: /^Cintel_\d{4}-\d{2}-\d{2}_(?:[A-Z0-9_-]+)$/i,
            confidence: 80 
        },
        
        // Blackmagic BMPCC/URSA (mit flexiblen Suffixen)
        // Matches: BMPCC_20220216, BMPCC_20220216_001, URSA_20220216_A001_BRAW, BMPC_240125_SCENE01
        { 
            vendor: 'Blackmagic', 
            rx: /^(?:BMPCC?|URSA)_\d{6,8}(?:_[A-Z0-9]{3,16})*$/i,
            confidence: 75 
        },
        
        // Canon Cinema EOS (mit optionalen Suffixen)
        // Matches: C001A001_20220216, C0001B0001_20220216_LOG, C123A456_240125_4K_HDR
        { 
            vendor: 'Canon', 
            rx: /^C\d{3,}[A-Z]?\d{3,}_\d{6,8}(?:_[A-Z0-9]{3,16})*$/i,
            confidence: 80 
        },
        
        // iPhone/Smartphone (mit Extension - spezifisch)
        // Matches: IMG_0001.MOV, IMG_1234.MP4
        { 
            vendor: 'iPhone/Smartphone', 
            rx: /^IMG_\d{4,}\.(?:MOV|MP4)$/i,
            confidence: 80 
        },
        
        // Smartphone (Datum/Zeit Format mit Extension)
        // Matches: VID_20220216_123045.MOV, VID_20250101_093000.MP4
        { 
            vendor: 'iPhone/Smartphone', 
            rx: /^VID_\d{8}_\d{6}\.(?:MOV|MP4)$/i,
            confidence: 85 
        },
        
        // Phantom Highspeed (mit flexiblen Suffixen)
        // Matches: P001234_C001, P001234_C001_20220216, Phantom_1234_C001, Phantom_1234_C001_HS, P000001_C001_2000FPS
        { 
            vendor: 'Phantom', 
            rx: /^(?:P\d{6,}_C\d{3,}|Phantom_\d+_C\d+)(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 85 
        },
        
        // VFX/Animation/Production (Projekt-Code Format)
        // Matches: PROJ_001_001, VFX_123_456, VFX_123_456_COMP, ANM_001_002_v3
        { 
            vendor: 'VFX/Production', 
            rx: /^[A-Z]{2,4}_\d{3,}_\d{3,}(?:_[A-Z0-9]+)*$/i,
            confidence: 70 
        },
        
        // VFX/Production (Scene/Shot/Take Format)
        // Matches: SC010_SH020_TK001, SC1_SH1_TK1, SC1_SH1_TK1_v2, SC100_SH050_TK003_FINAL
        { 
            vendor: 'VFX/Production', 
            rx: /^SC\d+_SH\d+_TK\d+(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 75 
        },
        
        // Generische Camera Stems (niedrige Confidence)
        // Matches: A001C001, B123C456, B123C456_v1 (ohne bekanntes Vendor-Pattern)
        { 
            vendor: 'Generic Camera Stem', 
            rx: /^[A-Z]\d{3,}C\d{3,}(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 60 
        },
        
        // Generische Camera Stems (mit Unterstrich-Prefix)
        // Matches: A_001C001, B_123C456, B_123C456_take1
        { 
            vendor: 'Generic Camera Stem', 
            rx: /^[A-Z]_\d{3,}C\d{3,}(?:_[A-Z0-9]{2,16})*$/i,
            confidence: 60 
        },
        
        // Generischer Fallback (kurze alphanumerische Codes)
        // Matches: A1234, ABC12345, XY123456Z
        { 
            vendor: 'Generic Fallback', 
            rx: /^[A-Z]{1,3}\d{4,8}[A-Z]?$/i,
            confidence: 50 
        },
        
        // Generischer Fallback (mit Media Extension)
        // Matches: Clip001.MXF, Take123.MOV, Shot456.R3D, File001.BRAW
        { 
            vendor: 'Generic Fallback', 
            rx: /^[A-Za-z]+\d+\.(?:MXF|MOV|MP4|R3D|BRAW)$/i,
            confidence: 55 
        }
    ],

    /**
     * Parse CMX 3600 EDL file and extract media references
     * @param {string} edlContent - The EDL content as string
     * @returns {Object} Object containing extracted media items
     */
    parseEDL: function(edlContent) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        try {
            var lines = edlContent.split(/\r?\n/);
            var mediaItems = [];
            var uniqueReels = {};
            var currentEvent = null;
            var eventCount = 0;
            var commentCount = 0;
            var skippedEvents = 0;
            var title = '';
            var allEvents = [];
            
            if (logger.isLogging) {
                logger.addLog('Starting EDL parse (' + lines.length + ' lines)');
            }
            
            console.log('Parsing EDL with', lines.length, 'lines');
            
            // PHASE 1: Collect all events and their comments
            for (var i = 0; i < lines.length; i++) {
                var line = lines[i].trim();
                
                if (!line) continue;
                
                // Capture title
                if (line.startsWith('TITLE:')) {
                    title = line.substring(6).trim();
                    if (logger.isLogging) {
                        logger.addLog('EDL title: ' + title);
                    }
                    continue;
                }
                
                if (line.startsWith('FCM:')) continue;
                
                // Parse event line
                var eventMatch = this.parseEventLine(line);
                if (eventMatch) {
                    // Check if this is a continuation of the current event (multi-line)
                    if (currentEvent && currentEvent.eventNumber === eventMatch.eventNumber) {
                        // Multi-line event - add to existing event
                        if (!currentEvent.multiLine) {
                            currentEvent.multiLine = [];
                        }
                        currentEvent.multiLine.push(eventMatch);
                        
                        if (logger.isLogging && debugMode) {
                            logger.addLog('  Multi-line continuation for event ' + eventMatch.eventNumber);
                        }
                    } else {
                        // New event
                        eventCount++;
                        
                        // Save previous event if exists
                        if (currentEvent) {
                            allEvents.push(currentEvent);
                        }
                        
                        // Initialize new event
                        currentEvent = {
                            eventNumber: eventMatch.eventNumber,
                            reelName: eventMatch.reelName,
                            trackType: eventMatch.trackType,
                            editType: eventMatch.editType,
                            sourceIn: eventMatch.sourceIn,
                            sourceOut: eventMatch.sourceOut,
                            recordIn: eventMatch.recordIn,
                            recordOut: eventMatch.recordOut,
                            sourceFile: null,
                            fromClipName: null,
                            locFileName: null,
                            comments: [],
                            multiLine: []
                        };
                    }
                    continue;
                } else if (line.match(/^\d+/)) {
                    // NEU: Zeile beginnt mit Nummer, sollte ein Event sein, aber Parsing fehlgeschlagen
                    if (logger.isLogging) {
                        logger.addLog('Failed to parse event line: "' + line + '"', 'WARNING');
                        
                        if (debugMode) {
                            // Character codes für debugging
                            var codes = [];
                            for (var c = 0; c < Math.min(line.length, 80); c++) {
                                codes.push(line.charCodeAt(c));
                            }
                            logger.addLog('  Char codes: ' + codes.join(','), 'WARNING');
                        }
                    }
                }
                
                // Collect comments for current event
                if (currentEvent && line.startsWith('*')) {
                    commentCount++;
                    currentEvent.comments.push(line);
                }
            }
            
            // Don't forget last event
            if (currentEvent) {
                allEvents.push(currentEvent);
            }
            
            // PHASE 2: Parse all comments for each event
            for (var j = 0; j < allEvents.length; j++) {
                var event = allEvents[j];
                
                // Parse all comments for this event
                for (var k = 0; k < event.comments.length; k++) {
                    var comment = event.comments[k];
                    
                    // Skip effect lines
                    if (comment.includes('EFFECT') || 
                        comment.includes('ASC_SOP') || 
                        comment.includes('ASC_SAT') ||
                        comment.includes('FREEZE FRAME') || 
                        comment.includes('TIMEWARP') ||
                        comment.includes('REPAIR:') ||
                        comment.includes('M2 ')) { // Skip M2 motion lines
                        continue;
                    }
                    
                    // Parse SOURCE FILE
                    if (comment.match(/\*\s*SOURCE\s+FILE:/i)) {
                        var sourceFileMatch = this.parseSourceFileDirective(comment);
                        if (sourceFileMatch) {
                            event.sourceFile = sourceFileMatch.sourceFile;
                            if (logger.isLogging && debugMode) {
                                logger.addLog('  Found SOURCE FILE: ' + sourceFileMatch.sourceFile);
                            }
                        } else {
                            // SOURCE FILE line found but value is NULL
                            event.sourceFile = false;
                        }
                        continue;
                    }
                    
                    // Parse FROM CLIP NAME
                    var fromClipMatch = this.parseFromClipNameComment(comment);
                    if (fromClipMatch) {
                        event.fromClipName = fromClipMatch.clipName;
                        if (logger.isLogging && debugMode) {
                            logger.addLog('  Found FROM CLIP NAME: ' + fromClipMatch.clipName);
                        }
                        continue;
                    }
                    
                    // Parse LOC comment
                    var locateMatch = this.parseLocateComment(comment);
                    if (locateMatch) {
                        event.locFileName = locateMatch.fileName;
                        if (logger.isLogging && debugMode) {
                            logger.addLog('  Found LOC: ' + locateMatch.fileName + 
                                        (locateMatch.isPathLocate ? ' (from path)' : ''));
                        }
                        continue;
                    }
                }
            }
            
            // PHASE 3: Process events with complete information
            for (var m = 0; m < allEvents.length; m++) {
                var eventToProcess = allEvents[m];
                
                if (this.shouldProcessEvent(eventToProcess)) {
                    this.processEvent(eventToProcess, uniqueReels);
                } else {
                    skippedEvents++;
                    if (logger.isLogging && debugMode) {
                        var reason = this.getSkipReason(eventToProcess);
                        logger.addLog('Event ' + eventToProcess.eventNumber + 
                                    ': Skipped (' + reason + ')', 'WARNING');
                    }
                }
            }
            
            mediaItems = Object.values(uniqueReels);
            
            // Log summary
            if (logger.isLogging) {
                logger.addLog('Processed ' + eventCount + ' events, ' + commentCount + ' comments, ' + 
                            skippedEvents + ' skipped');
                
                if (mediaItems.length > 0) {
                    var sourceCounts = {};
                    mediaItems.forEach(function(item) {
                        sourceCounts[item.source] = (sourceCounts[item.source] || 0) + 1;
                    });
                    var summary = Object.keys(sourceCounts).map(function(source) {
                        return source + ': ' + sourceCounts[source];
                    }).join(', ');
                    logger.addLog('Found ' + mediaItems.length + ' unique files (' + summary + ')');
                }
            }
            
            console.log('EDL parsing completed:', mediaItems.length, 'unique media items found');
            
            return {
                success: true,
                mediaItems: mediaItems,
                totalCount: mediaItems.length,
                formatType: 'EDL',
                detectedFormat: 'CMX 3600 EDL'
            };
            
        } catch (error) {
            if (logger.isLogging) {
                logger.addLog('EDL parse failed - ' + error.message, 'ERROR');
            }
            console.error('Error parsing EDL content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    /**
     * Get skip reason for debugging
     */
    getSkipReason: function(event) {
        // Check if SOURCE FILE is explicitly NULL
        if (event.sourceFile === false) {
            return 'source file is NULL';
        }
        
        // Check SOURCE FILE confidence
        if (event.sourceFile) {
            var normalized = this.normalizeFileName(event.sourceFile);
            if (!normalized) {
                return 'source file normalization failed';
            }
            var score = this.getFileNameConfidence(normalized);
            if (score < 50) {
                return 'source file confidence too low (' + score + ')';
            }
        }
        
        // Check technical reels
        var technicalReels = ['BLACK', 'SLUG', 'BARS', 'TONE'];
        if (technicalReels.indexOf(event.reelName.toUpperCase()) !== -1) {
            return 'technical reel: ' + event.reelName + ' (no valid media files)';
        }
        
        // Check BL reel
        if (event.reelName.toUpperCase() === 'BL' && !event.fromClipName) {
            return 'BL without clip name';
        }
        
        // Check edit type
        if (event.editType !== 'C' && !event.sourceFile) {
            return 'non-cut without source file';
        }
        
        // Check no viable filename
        var result = this.getBestFileName(event);
        if (!result) {
            return 'no viable filename found (all candidates below threshold)';
        }
        
        return 'reel: ' + event.reelName + ', type: ' + event.editType;
    },

    /**
     * Check if event should be processed
     */
    shouldProcessEvent: function(event) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // SOURCE FILE explizit NULL? Skip
        if (event.sourceFile === false) {
            if (logger.isLogging) {
                logger.addLog('Event ' + event.eventNumber + ': SOURCE FILE is NULL, checking alternatives');
            }
            
            // Prüfe ob andere Quellen gut genug sind
            var hasValidAlternative = false;
            
            if (event.fromClipName) {
                var fromNormalized = this.normalizeFileName(event.fromClipName);
                if (fromNormalized) {
                    var fromScore = this.getFileNameConfidence(fromNormalized);
                    if (fromScore >= 70) { // Hoher Threshold für NULL-Alternativen
                        hasValidAlternative = true;
                    }
                }
            }
            
            if (!hasValidAlternative && event.locFileName) {
                var locNormalized = this.normalizeFileName(event.locFileName);
                if (locNormalized) {
                    var locScore = this.getFileNameConfidence(locNormalized);
                    if (locScore >= 70) {
                        hasValidAlternative = true;
                    }
                }
            }
            
            if (!hasValidAlternative) {
                if (logger.isLogging) {
                    logger.addLog('Event ' + event.eventNumber + ': SOURCE FILE is NULL and no valid alternatives');
                }
                return false;
            }
        }
        
        // Technical Reels special handling
        var technicalReels = ['BLACK', 'SLUG', 'BARS', 'TONE'];
        if (technicalReels.indexOf(event.reelName.toUpperCase()) !== -1) {
            if (logger.isLogging) {
                logger.addLog('Event ' + event.eventNumber + ': Technical reel "' + event.reelName + 
                            '", checking for valid media files');
            }
            
            // Prüfe alle Quellen für technical reels
            var result = this.getBestFileName(event);
            if (result && this.getFileNameConfidence(result.fileName) >= 70) {
                if (logger.isLogging) {
                    logger.addLog('  → Accepting technical reel due to valid media file');
                }
                return true;
            }
            
            if (logger.isLogging) {
                logger.addLog('  → Rejecting technical reel - no valid media files found');
            }
            return false;
        }
        
        // Für alle anderen Events: getBestFileName entscheidet
        var result = this.getBestFileName(event);
        if (result) {
            if (logger.isLogging && debugMode) {
                logger.addLog('Event ' + event.eventNumber + ': Found valid candidate "' + 
                             result.fileName + '" from ' + result.source);
            }
            return true;
        }
        
        if (logger.isLogging && debugMode) {
            logger.addLog('Event ' + event.eventNumber + ': No valid filename found, skipping');
        }
        return false;
    },

    /**
     * Parse main event line
     */
    parseEventLine: function(line) {
        var eventRegex = /^(\d+)\s+([^\s]+)\s+([VAva]+\d*)\s+([^\s]+)\s+([\d:]+)\s+([\d:]+)\s+([\d:]+)\s+([\d:]+)/;
        var match = line.match(eventRegex);
        
        if (match) {
            return {
                eventNumber: parseInt(match[1]),
                reelName: match[2],
                trackType: match[3],
                editType: match[4],
                sourceIn: match[5],
                sourceOut: match[6],
                recordIn: match[7],
                recordOut: match[8]
            };
        }
        
        return null;
    },

    /**
     * Parse SOURCE FILE directive
     */
    parseSourceFileDirective: function(line) {
        var sourceFileRegex = /\*\s*SOURCE\s+FILE:\s*(.+)/i;
        var match = line.match(sourceFileRegex);
        
        if (match) {
            var sourceFile = match[1].trim();
            
            // Case-insensitive NULL check
            var upperCase = sourceFile.toUpperCase();
            if (upperCase === '(NULL)' || 
                upperCase === 'NULL' || 
                upperCase === '(NULL)' ||
                upperCase === '[NULL]' ||
                upperCase === '"NULL"' ||
                upperCase === "'NULL'") {
                
                var logger = P5Index.util.ParserLogger;
                if (logger.isLogging) {
                    logger.addLog('  SOURCE FILE is NULL variant: ' + sourceFile, 'WARNING');
                }
                return null;
            }
            
            return {
                sourceFile: sourceFile
            };
        }
        
        return null;
    },

    /**
     * Parse FROM CLIP NAME comment
     */
    parseFromClipNameComment: function(line) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Log wenn die Zeile FROM CLIP NAME enthält
        if (debugMode && logger.isLogging && line.includes('FROM CLIP NAME')) {
            logger.addLog('  Checking FROM CLIP NAME line: "' + line + '"');
            // Optional: Character codes für Debug
            var codes = [];
            for (var i = 0; i < Math.min(line.length, 50); i++) {
                codes.push(line.charCodeAt(i));
            }
            logger.addLog('  First 50 char codes: ' + codes.join(','));
        }
        
        var fromClipRegex = /\*\s*FROM\s+CLIP\s+NAME:\s*(.+)/i;
        var match = line.match(fromClipRegex);
        
        if (match) {
            return {
                clipName: match[1].trim()
            };
        } else if (debugMode && logger.isLogging && line.includes('FROM CLIP NAME')) {
            // NEU: Regex Match fehlgeschlagen, aber FROM CLIP NAME war in der Zeile
            logger.addLog('  FROM CLIP NAME regex failed to match!', 'WARNING');
        }
        
        return null;
    },

    /**
     * Parse LOC comment
     */
    parseLocateComment: function(line) {
        // Path pattern (for full paths)
        var pathLocateRegex = /\*\s*LOC:\s*([A-Z]:[\\/].*|\/.*)/i;
        var pathMatch = line.match(pathLocateRegex);
        
        if (pathMatch) {
            var fullPath = pathMatch[1].trim();
            var fileName = this.getFileName(fullPath);
            
            var logger = P5Index.util.ParserLogger;
            if (logger.isLogging) {
                logger.addLog('  LOC path detected: ' + fullPath + ' -> ' + fileName);
            }
            
            // Handle case where path has extra text after
            var parts = fullPath.split(/\s+/);
            if (parts.length > 1) {
                // Take first part as path
                fileName = this.getFileName(parts[0]);
                if (logger.isLogging) {
                    logger.addLog('  LOC path has extra text, using: ' + fileName);
                }
            }
            
            return {
                fileName: fileName,
                fullPath: fullPath,
                isPathLocate: true
            };
        }
        
        // VFX pattern
        var vfxLocateRegex = /\*\s*LOC:\s*([\d:]+)\s+(\w+)\s+(.+)/i;
        var vfxMatch = line.match(vfxLocateRegex);
        
        if (vfxMatch) {
            var fileName = vfxMatch[3].trim();
            // Skip VFX placeholders
            if (/^[A-Z]+_\d+_\d+_\d+_PL\d+_v\d+$/i.test(fileName)) {
                return null;
            }
            return {
                fileName: fileName,
                locateTime: vfxMatch[1],
                color: vfxMatch[2],
                isVFXLocate: true
            };
        }
        
        // Simple pattern
        var simpleLocateRegex = /\*\s*LOC:\s*([\d:]+)\s+(.+)/i;
        var simpleMatch = line.match(simpleLocateRegex);
        
        if (simpleMatch) {
            return {
                fileName: simpleMatch[2].trim(),
                locateTime: simpleMatch[1],
                isVFXLocate: false
            };
        }
        
        return null;
    },

    /**
     * Normalize filename by removing quotes and extracting from paths
     */
    normalizeFileName: function(fileName) {
        if (!fileName) return fileName;
        
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        var original = fileName;
        
        // Timeline-Positionen ablehnen
        if (/^\d+\/\d+\/\d+$/.test(fileName)) {
            if (logger.isLogging && debugMode) {
                logger.addLog('  Rejected timeline position: "' + fileName + '"');
            }
            return null;
        }
        
        // Remove surrounding quotes and whitespace
        fileName = fileName.replace(/^[\s"'""„‟‚''«»\u201C\u201D]+|[\s"'""„‟‚''«»\u201C\u201D]+$/g, '');
        
        // Extract filename from Windows path
        if (fileName.includes('\\')) {
            fileName = fileName.split('\\').pop();
        }
        
        // Extract filename from Unix path
        if (fileName.includes('/')) {
            fileName = fileName.split('/').pop();
        }
        
        // Remove any remaining leading/trailing whitespace
        fileName = fileName.trim();
        
        if (logger.isLogging && debugMode && original !== fileName) {
            logger.addLog('  Normalized: "' + original + '" -> "' + fileName + '"');
        }
        
        return fileName;
    },

    /**
     * Bestimme den besten Dateinamen für ein Event
     */
    getBestFileName: function(event) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // PRIORITÄT 1: SOURCE FILE (wenn vorhanden und valid)
        if (event.sourceFile && event.sourceFile !== false) {
            var normalized = this.normalizeFileName(event.sourceFile);
            if (normalized) {
                var sourceScore = this.getFileNameConfidence(normalized);
                if (sourceScore >= 50) {  // Threshold für SOURCE FILE
                    if (logger.isLogging && debugMode) {
                        logger.addLog('Using SOURCE FILE: ' + normalized + ' (score: ' + sourceScore + ')');
                    }
                    return {
                        fileName: normalized,
                        source: 'source_file'
                    };
                } else if (logger.isLogging && debugMode) {
                    logger.addLog('SOURCE FILE confidence too low: ' + normalized + ' (score: ' + sourceScore + ')');
                }
            }
        }
        
        // PRIORITÄT 2: Intelligente Auswahl aus anderen Quellen
        var candidates = [];
        
        if (event.fromClipName) {
            var fromNormalized = this.normalizeFileName(event.fromClipName);
            if (fromNormalized) {
                candidates.push({
                    value: fromNormalized,
                    source: 'from_clip_name',
                    score: this.getFileNameConfidence(fromNormalized)
                });
            }
        }
        
        if (event.locFileName) {
            var locNormalized = this.normalizeFileName(event.locFileName);
            if (locNormalized) {
                candidates.push({
                    value: locNormalized,
                    source: 'loc',
                    score: this.getFileNameConfidence(locNormalized)
                });
            }
        }
        
        // Reel name nur als Kandidat wenn es kein technischer Reel ist
        if (event.reelName && !['AX', 'BL', 'BLACK', 'SLUG', 'BARS', 'TONE'].includes(event.reelName.toUpperCase())) {
            candidates.push({
                value: event.reelName,
                source: 'reel',
                score: this.getFileNameConfidence(event.reelName)
            });
        }
        
        // Sortiere nach Score
        candidates.sort(function(a, b) {
            // Bei gleichem Score: Länge als Tie-Breaker
            if (b.score === a.score) {
                return b.value.length - a.value.length;
            }
            return b.score - a.score;
        });
        
        // Log candidate scores in debug mode
        if (logger.isLogging && debugMode && candidates.length > 0) {
            logger.addLog('Event ' + event.eventNumber + ' candidates:');
            candidates.forEach(function(c) {
                logger.addLog('  - ' + c.source + ': "' + c.value + '" (score: ' + c.score + ')');
            });
        }
        
        // Wähle besten Kandidaten mit minimalem Score
        if (candidates.length > 0 && candidates[0].score >= 15) {  // Minimaler Threshold
            if (logger.isLogging && debugMode) {
                logger.addLog('Best candidate: ' + candidates[0].value + 
                            ' from ' + candidates[0].source + 
                            ' (score: ' + candidates[0].score + ')');
            }
            return {
                fileName: candidates[0].value,
                source: candidates[0].source
            };
        }
        
        if (logger.isLogging && debugMode) {
            logger.addLog('Event ' + event.eventNumber + ': No candidate met minimum score threshold');
        }
        
        return null;
    },

    /**
     * Konfidenz-Score für Dateinamen
     */
    getFileNameConfidence: function(str) {
        if (!str || typeof str !== 'string') return 0;
        
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        var score = 0;
        var reason = '';
        
        // NEU: Kommas sind fast sicher KEINE Dateinamen
        if (str.includes(',')) {
            score = 0;
            reason = 'contains comma (likely description/metadata)';
            
            if (logger.isLogging && debugMode) {
                logger.addLog('    Confidence for "' + str + '": ' + score + ' (' + reason + ')');
            }
            return 0;
        }
        
        // MIT EXTENSION - dann sind Spaces und Punkte erlaubt
        if (/\.[a-zA-Z0-9]{2,5}$/.test(str)) {
            var ext = str.split('.').pop().toLowerCase();
            
            // Media-Extensions
            if (/^(mxf|mov|mp4|m4v|avi|r3d|dpx|exr|wav|aiff|mp3|m4a|braw|ari|cri)$/i.test(ext)) {
                score = 90;
                reason = 'media extension';
            }
            // Bild/Dokument-Extensions
            else if (/^(tif|tiff|jpg|jpeg|png|heic|pdf|xml|aaf)$/i.test(ext)) {
                score = 75;
                reason = 'image/doc extension';
            }
            // Projekt-Extensions (niedriger Score)
            else if (/^(prproj|aep|drp|comp|nk|hip|blend|c4d)$/i.test(ext)) {
                score = 40;
                reason = 'project file extension';
            }
            // Text/Config (sehr niedrig)
            else if (/^(txt|log|cfg|ini|json|csv)$/i.test(ext)) {
                score = 25;
                reason = 'text/config extension';
            }
            // Andere Extensions
            else {
                score = 70;
                reason = 'other extension';
            }
            
            if (logger.isLogging && debugMode) {
                logger.addLog('    Confidence for "' + str + '": ' + score + ' (' + reason + ')');
            }
            return score;
        }
        
        // AB HIER: OHNE EXTENSION
        // Jetzt sind Spaces verdächtig!
        
        // NEU: Strings mit Spaces OHNE Extension = keine Filenames
        if (/\s/.test(str)) {
            score = 0;
            reason = 'contains spaces without extension (likely description/metadata)';
            
            if (logger.isLogging && debugMode) {
                logger.addLog('    Confidence for "' + str + '": ' + score + ' (' + reason + ')');
            }
            return 0;
        }
        
        // NEU: Mehrere Punkte OHNE Extension am Ende = verdächtig
        if (str.includes('.')) {
            var parts = str.split('.');
            // Wenn wir hier sind, hat der String KEINE valide Extension (sonst wären wir schon returned)
            // Aber mehrere Punkte ohne Extension ist verdächtig
            if (parts.length > 2) {
                score = 0;
                reason = 'multiple dots without valid extension (likely description)';
                
                if (logger.isLogging && debugMode) {
                    logger.addLog('    Confidence for "' + str + '": ' + score + ' (' + reason + ')');
                }
                return 0;
            }
        }
        
        // OHNE EXTENSION - Pattern-Analyse
        
        // Definitiv KEINE Dateinamen
        if (/^\d+\/\d+\/\d+$/.test(str)) {
            score = 0;
            reason = 'timeline position';
        }
        else if (/^(AX|BL|BLACK|SLUG|BARS|TONE)$/i.test(str)) {
            score = 0;
            reason = 'technical reel';
        }
        else if (/FAUX-CLIP/i.test(str)) {
            score = 0;
            reason = 'faux clip';
        }
        else if (/^(Solid Color|Cross Dissolve|Adjustment Clip|Transition)$/i.test(str)) {
            score = 0;
            reason = 'effect name';
        }
        else if (str.length < 4) {
            score = 10;
            reason = 'too short';
        }
        else {
            // Camera Pattern Check
            var cameraMatch = false;
            for (var i = 0; i < this.CAMERA_PATTERNS.length; i++) {
                var pattern = this.CAMERA_PATTERNS[i];
                if (pattern.rx.test(str)) {
                    score = pattern.confidence;
                    reason = pattern.vendor + ' pattern [#' + i + ']';  // NEU: Pattern-Index
                    cameraMatch = true;
                    break;
                }
            }
            
            if (!cameraMatch) {
                // Single word check NACH Camera Pattern Check
                var isSingleWord = !str.includes('.') && 
                                   !str.includes('_') && 
                                   !str.includes('-') && 
                                   !/\d/.test(str);
                
                if (isSingleWord) {
                    score = 0;
                    reason = 'single word without numbers (likely effect name)';
                }
                // Generische Patterns (nur wenn nicht single word)
                else if (/^[A-Za-z0-9][A-Za-z0-9_\-\.]{6,}[A-Za-z0-9]$/.test(str)) {
                    score = 40;
                    reason = 'generic filename pattern';
                }
                else if (/^[A-Z0-9_]{4,12}$/.test(str)) {
                    score = 35;
                    reason = 'short alphanumeric';
                }
                else {
                    score = 20;
                    reason = 'default/unknown';
                }
            }
        }
        
        // Längen-Bonus für sehr lange Strings (aber begrenzt)
        if (score >= 35 && str.length > 20) {
            var lengthBonus = Math.min(Math.floor((str.length - 20) / 5), 10);
            if (lengthBonus > 0) {
                score += lengthBonus;
                reason += ' +length';
            }
        }
        
        if (logger.isLogging && debugMode) {
            logger.addLog('    Confidence for "' + str + '": ' + score + ' (' + reason + ')');
        }
        
        return score;
    },

    /**
     * Process event and add to unique reels
     */
    processEvent: function(event, uniqueReels) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        // Bestimme besten Dateinamen
        var result = this.getBestFileName(event);
        
        if (!result) {
            if (logger.isLogging && debugMode) {
                logger.addLog('Event ' + event.eventNumber + ': No usable filename found, skipped', 'WARNING');
            }
            return;
        }
        
        var fileName = result.fileName;
        var key = fileName.toLowerCase();
        
        if (!uniqueReels[key]) {
            uniqueReels[key] = {
                fileName: fileName,
                path: this.generatePath(fileName, fileName),
                usedDurationTC: this.calculateDuration(event.sourceIn, event.sourceOut),
                availableDurationTC: '',
                source: result.source,
                originalReelName: event.reelName,
                editType: event.editType,
                events: []
            };
            
            if (logger.isLogging) {
                logger.addLog('Event ' + event.eventNumber + ': Added "' + fileName + 
                            '" from ' + result.source);
            }
        } else {
            if (logger.isLogging && debugMode) {
                logger.addLog('Event ' + event.eventNumber + ': File "' + fileName + 
                            '" already exists, adding to events');
            }
        }
        
        // Event info hinzufügen
        uniqueReels[key].events.push({
            eventNumber: event.eventNumber,
            sourceIn: event.sourceIn,
            sourceOut: event.sourceOut,
            recordIn: event.recordIn,
            recordOut: event.recordOut,
            reelName: event.reelName,
            editType: event.editType,
            multiLine: event.multiLine ? true : false
        });
    },

    /**
     * Generate path from filename
     */
    generatePath: function(originalPath, fileName) {
        // EDL files typically don't contain paths, just filenames
        return fileName;
    },

    /**
     * Calculate duration from timecodes
     */
    calculateDuration: function(inTime, outTime) {
        try {
            var inFrames = this.timecodeToFrames(inTime);
            var outFrames = this.timecodeToFrames(outTime);
            var durationFrames = outFrames - inFrames;
            
            if (durationFrames > 0) {
                return this.framesToTimecode(durationFrames);
            }
        } catch (e) {
            console.warn('Could not calculate duration for', inTime, '-', outTime);
        }
        
        return '';
    },

    /**
     * Timecode to frames conversion
     */
    timecodeToFrames: function(timecode, fps) {
        fps = fps || 25;
        
        var parts = timecode.split(':');
        if (parts.length !== 4) return 0;
        
        var hours = parseInt(parts[0]) || 0;
        var minutes = parseInt(parts[1]) || 0;
        var seconds = parseInt(parts[2]) || 0;
        var frames = parseInt(parts[3]) || 0;
        
        return (hours * 3600 + minutes * 60 + seconds) * fps + frames;
    },

    /**
     * Frames to timecode conversion
     */
    framesToTimecode: function(frames, fps) {
        fps = fps || 25;
        
        var totalSeconds = Math.floor(frames / fps);
        var h = Math.floor(totalSeconds / 3600);
        var m = Math.floor((totalSeconds % 3600) / 60);
        var s = totalSeconds % 60;
        var f = frames % fps;
        
        return h.toString().padStart(2, '0') + ':' + 
               m.toString().padStart(2, '0') + ':' + 
               s.toString().padStart(2, '0') + ':' + 
               f.toString().padStart(2, '0');
    },

    /**
     * Extract filename from path
     */
    getFileName: function(path) {
        if (!path) return '';
        
        // Handle both Windows and Unix paths
        var fileName = path.split('\\').pop().split('/').pop();
        
        // Log if we extracted from a path
        var logger = P5Index.util.ParserLogger;
        if (logger.isLogging && fileName !== path) {
            logger.addLog('  Extracted filename: "' + path + '" -> "' + fileName + '"');
        }
        
        return fileName;
    }
});