Ext.define('P5Index.util.OTIOParser', {
    singleton: true,
    debugMode: false,

    /**
     * Parse OTIO file and extract media references
     * @param {string} otioContent - The OTIO JSON content as string
     * @returns {Object} Object containing extracted media items
     */
    parseOTIO: function(otioContent) {
        var logger = P5Index.util.ParserLogger;
        var debugMode = this.debugMode;
        
        try {
            // Log parse start
            if (logger.isLogging) {
                logger.addLog('Starting OTIO parse');
            }
            
            // Parse JSON content
            const otioData = JSON.parse(otioContent);
            
            // Analyze OTIO structure and log
            if (logger.isLogging) {
                var rootSchema = otioData.OTIO_SCHEMA || 'unknown';
                var timelineSchema = 'unknown';
                var trackCount = 0;
                
                if (!otioData.OTIO_SCHEMA) {
                    logger.addLog('No OTIO_SCHEMA found - invalid OTIO file?', 'WARNING');
                }
                
                if (otioData.tracks) {
                    timelineSchema = otioData.tracks.OTIO_SCHEMA || 'unknown';
                    trackCount = otioData.tracks.children?.length || 0;
                } else {
                    logger.addLog('No tracks found in OTIO structure', 'WARNING');
                }
                
                logger.addLog('Root schema: ' + rootSchema + ', Timeline schema: ' + timelineSchema);
                logger.addLog('Found ' + trackCount + ' tracks');
            }
            
            // Result array for media paths
            const mediaItems = [];
            var trackClipCounts = [];
            
            // Process tracks and clips
            if (otioData.tracks && otioData.tracks.children) {
                const tracks = otioData.tracks.children;
                
                for (let trackIndex = 0; trackIndex < tracks.length; trackIndex++) {
                    const track = tracks[trackIndex];
                    if (track.children) {
                        const clips = track.children;
                        trackClipCounts.push(clips.length);
                        
                        for (let clipIndex = 0; clipIndex < clips.length; clipIndex++) {
                            const clip = clips[clipIndex];
                            
                            // Log JEDEN Clip-Typ für Debugging
                            if (logger.isLogging && debugMode) {
                                if (clip.OTIO_SCHEMA !== "Clip.2") {
                                    logger.addLog('Track ' + trackIndex + ', item ' + clipIndex + 
                                                ': Skipped ' + (clip.OTIO_SCHEMA || 'unknown') + 
                                                ' "' + (clip.name || 'unnamed') + '"', 'WARNING');
                                }
                            }
                            
                            // Only process clips (not gaps)
                            if (clip.OTIO_SCHEMA === "Clip.2") {
                                if (!clip.media_references || !clip.media_references.DEFAULT_MEDIA) {
                                    if (logger.isLogging) {
                                        logger.addLog('Track ' + trackIndex + ', clip ' + clipIndex + 
                                                    ': No media reference for "' + clip.name + '"', 'WARNING');
                                    }
                                    continue;
                                }
                                
                                const mediaRef = clip.media_references.DEFAULT_MEDIA;
                                const mediaSchema = mediaRef.OTIO_SCHEMA;
                                
                                // Log media reference type
                                if (logger.isLogging && debugMode) {
                                    logger.addLog('Processing ' + mediaSchema + ': "' + clip.name + '"');
                                }
                                
                                if (mediaSchema === "ImageSequenceReference.1") {
                                    // ARRI RAW oder ähnliche Image Sequences
                                    const baseUrl = mediaRef.target_url_base;
                                    const prefix = mediaRef.name_prefix;
                                    const suffix = mediaRef.name_suffix;
                                    const padding = mediaRef.frame_zero_padding;
                                    
                                    // Available range (full sequence)
                                    const startFrame = mediaRef.start_frame;
                                    const availableDuration = mediaRef.available_range.duration.value;
                                    const availableRate = mediaRef.available_range.duration.rate || 25;
                                    const endFrame = startFrame + availableDuration - 1;
                                    
                                    // Used range (cut range if available)
                                    let usedDuration = availableDuration;
                                    let usedRate = availableRate;
                                    
                                    if (clip.source_range) {
                                        usedDuration = clip.source_range.duration.value;
                                        usedRate = clip.source_range.duration.rate || availableRate;
                                    }
                                    
                                    // FIX: Base name für Archive-Suche extrahieren
                                    const baseName = prefix.replace(/\.$/, '');
                                    
                                    // FIX: Verwende Folder-Pfad statt einzelnen Frame
                                    const containerPath = baseUrl;
                                    
                                    // Optional: Ersten Frame für Preview behalten
                                    const firstFrameNum = startFrame.toString().padStart(padding, '0');
                                    const firstFramePath = baseUrl + '/' + prefix + firstFrameNum + suffix;
                                    
                                    if (logger.isLogging && debugMode) {
                                        logger.addLog('  → Image sequence: ' + baseName + 
                                                    ' (' + endFrame - startFrame + 1 + ' frames)');
                                    }
                                    
                                    // Store information optimized für Archive-Suche
                                    mediaItems.push({
                                        type: "imageSequence",
                                        fileName: baseName,
                                        path: containerPath,
                                        originalFile: clip.name,
                                        displayName: clip.name + " (sequence)",
                                        usedDurationTC: this.formatTimecode(usedDuration, usedRate),
                                        availableDurationTC: this.formatTimecode(availableDuration, availableRate),
                                        baseUrl: baseUrl,
                                        namePrefix: prefix,
                                        nameSuffix: suffix,
                                        padding: padding,
                                        fullStartFrame: startFrame,
                                        fullEndFrame: endFrame,
                                        frameCount: endFrame - startFrame + 1,
                                        previewPath: firstFramePath
                                    });
                                    
                                } else if (mediaSchema === "ExternalReference.1") {
                                    // MXF or other single files
                                    if (!mediaRef.target_url) {
                                        if (logger.isLogging) {
                                            logger.addLog('  → No target_url for "' + clip.name + '"', 'WARNING');
                                        }
                                        continue;
                                    }
                                    
                                    const fullPath = mediaRef.target_url;
                                    const fileName = this.getFileName(fullPath);
                                    const dirPath = this.getDirPath(fullPath);
                                    
                                    // Available range (full file duration)
                                    const availableStartTime = mediaRef.available_range.start_time.value;
                                    const availableDuration = mediaRef.available_range.duration.value;
                                    const availableRate = mediaRef.available_range.duration.rate || 25;
                                    
                                    // Used range (cut range if available)
                                    let usedDuration = availableDuration;
                                    let usedRate = availableRate;
                                    
                                    if (clip.source_range) {
                                        usedDuration = clip.source_range.duration.value;
                                        usedRate = clip.source_range.duration.rate || availableRate;
                                    }
                                    
                                    // Check for special media types
                                    let mediaType = "singleFile";
                                    let containerPath = "";
                                    let displayName = fileName;
                                    
                                    // 1. RED material (.R3D and .RDC folder)
                                    if (fileName.match(/\.R3D$/i)) {
                                        mediaType = "redFolder";
                                        // Extract RDC folder path
                                        if (this.getFileName(dirPath).match(/\.RDC$/i)) {
                                            containerPath = dirPath;
                                            displayName = this.getFileName(dirPath);
                                        }
                                    }
                                    
                                    // 2. SONY material (MXF in clip folder structure)
                                    if (fileName.match(/\.mxf$/i)) {
                                        // Check SONY F55/F65/FX9 folder structure
                                        const parentDir = this.getFileName(dirPath);
                                        if (parentDir === fileName || 
                                            this.trimExtension(parentDir, ".mxf") === this.trimExtension(fileName, ".mxf")) {
                                            mediaType = "sonyClipFolder";
                                            containerPath = dirPath;
                                            displayName = parentDir;
                                        }
                                    }
                                    
                                    // 3. Phantom CINE material
                                    if (fileName.match(/\.cine$/i)) {
                                        mediaType = "cineFile";
                                    }
                                    
                                    // Log detection result
                                    if (logger.isLogging && debugMode) {
                                        if (mediaType !== "singleFile") {
                                            logger.addLog('  → Detected as ' + mediaType + ': ' + displayName);
                                        }
                                    }
                                    
                                    // Base media item with OTIO-conform fields
                                    const mediaItem = {
                                        type: mediaType,
                                        fileName: fileName,
                                        path: mediaType === "singleFile" ? fullPath : containerPath,
                                        originalFile: fullPath,
                                        displayName: displayName,
                                        usedDurationTC: this.formatTimecode(usedDuration, usedRate),
                                        availableDurationTC: this.formatTimecode(availableDuration, availableRate)
                                    };
                                    
                                    // Add additional properties based on media type
                                    if (mediaType === "redFolder" || mediaType === "sonyClipFolder") {
                                        mediaItem.containerPath = containerPath;
                                    }
                                    
                                    mediaItems.push(mediaItem);
                                    
                                } else if (mediaSchema === "MissingReference.1") {
                                    // Missing/Offline Media - aber Dateiname für Archive-Suche extrahieren
                                    const fileName = mediaRef.metadata?.AAF?.Name || clip.name;
                                    
                                    if (!fileName) {
                                        if (logger.isLogging) {
                                            logger.addLog('  → Missing reference without name, skipped', 'WARNING');
                                        }
                                        continue;
                                    }
                                    
                                    if (logger.isLogging && debugMode) {
                                        logger.addLog('  → Missing/offline: ' + fileName);
                                    }
                                    
                                    // Available range (wenn vorhanden)
                                    const availableDuration = mediaRef.available_range?.duration?.value || 0;
                                    const availableRate = mediaRef.available_range?.duration?.rate || 25;
                                    
                                    // Used range from clip
                                    let usedDuration = availableDuration;
                                    let usedRate = availableRate;
                                    
                                    if (clip.source_range) {
                                        usedDuration = clip.source_range.duration.value;
                                        usedRate = clip.source_range.duration.rate || availableRate;
                                    }
                                    
                                    mediaItems.push({
                                        type: "missingFile",
                                        fileName: fileName,
                                        path: fileName,
                                        originalFile: fileName,
                                        displayName: fileName + " (offline)",
                                        usedDurationTC: this.formatTimecode(usedDuration, usedRate),
                                        availableDurationTC: this.formatTimecode(availableDuration, availableRate)
                                    });
                                    
                                } else {
                                    // Unbekannter Media Reference Type
                                    if (logger.isLogging) {
                                        logger.addLog('  → Unknown media schema: ' + mediaSchema + 
                                                    ' for "' + clip.name + '"', 'WARNING');
                                    }
                                }
                            }
                        }
                    }
                }
                
                // Log track structure
                if (logger.isLogging && trackClipCounts.length > 0) {
                    logger.addLog('Track structure: ' + trackClipCounts.join(', ') + ' clips');
                }
            }
            
            // Deduplizierung mit Details
            if (logger.isLogging && debugMode && mediaItems.length > 0) {
                logger.addLog('Starting deduplication of ' + mediaItems.length + ' items');
            }
            
            // Remove duplicates (same file can appear multiple times)
            const uniquePaths = {};
            const duplicates = [];
            
            for (const item of mediaItems) {
                const type = item.type;
                let key = "";
                
                switch (type) {
                    case "imageSequence":
                        // For image sequences, base path is the key
                        key = item.baseUrl || item.path;
                        break;
                    case "redFolder":
                    case "sonyClipFolder":
                        // For folders, folder path is the key
                        key = item.path;
                        break;
                    default:
                        // For single files, full path is the key
                        key = item.path;
                        break;
                }
                
                if (key !== "") {
                    if (!uniquePaths[key]) {
                        uniquePaths[key] = item;
                    } else {
                        // Log duplicates
                        duplicates.push({
                            kept: uniquePaths[key].displayName || uniquePaths[key].fileName,
                            removed: item.displayName || item.fileName,
                            reason: 'duplicate path'
                        });
                        
                        // Duration comparison - keep longer one
                        if (this.parseTimecode(item.usedDurationTC) > this.parseTimecode(uniquePaths[key].usedDurationTC)) {
                            duplicates[duplicates.length-1].reason = 'shorter duration';
                            uniquePaths[key] = item;
                        }
                    }
                }
            }
            
            // Create unique list
            const uniqueItems = Object.values(uniquePaths);
            
            // Log deduplizierungs-Details
            if (logger.isLogging && duplicates.length > 0) {
                if (debugMode) {
                    // Erste paar Duplicates zeigen
                    duplicates.slice(0, 3).forEach(function(dup) {
                        logger.addLog('Duplicate removed: ' + dup.removed + ' (' + dup.reason + ')');
                    });
                    if (duplicates.length > 3) {
                        logger.addLog('... and ' + (duplicates.length - 3) + ' more duplicates');
                    }
                }
            }
            
            // Log deduplification info
            if (logger.isLogging) {
                var totalClips = mediaItems.length;
                var uniqueCount = uniqueItems.length;
                
                if (totalClips !== uniqueCount) {
                    logger.addLog('Deduplicated: ' + totalClips + ' clips → ' + uniqueCount + ' unique files');
                }
                
                // Log media type summary
                if (uniqueItems.length > 0) {
                    var typeCounts = {};
                    uniqueItems.forEach(function(item) {
                        typeCounts[item.type] = (typeCounts[item.type] || 0) + 1;
                    });
                    var summary = Object.keys(typeCounts).map(function(type) {
                        return type + ': ' + typeCounts[type];
                    }).join(', ');
                    logger.addLog('Media types: ' + summary);
                }
            }
            
            // Return the results
            return {
                success: true,
                mediaItems: uniqueItems,
                totalCount: uniqueItems.length,
                formatType: 'OTIO',
                detectedFormat: 'OpenTimelineIO (OTIO)',
                metadata: {
                    timelineCount: mediaItems.length,
                    dedupedCount: uniqueItems.length,
                    duplicatesRemoved: mediaItems.length - uniqueItems.length
                }
            };
            
        } catch (error) {
            // Log parse error
            if (logger.isLogging) {
                logger.addLog('OTIO parse failed - ' + error.message, 'ERROR');
                if (error.message.includes('JSON')) {
                    logger.addLog('Invalid JSON structure in OTIO file', 'ERROR');
                }
            }
            
            console.error('Error parsing OTIO content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    },

    /**
     * Extract filename from path
     * @param {string} path - File path
     * @returns {string} Filename
     */
    getFileName: function(path) {
        return path.split('/').pop().split('\\').pop();
    },

    /**
     * Extract directory path from full path
     * @param {string} path - File path
     * @returns {string} Directory path
     */
    getDirPath: function(path) {
        const parts = path.split(/[\/\\]/);
        parts.pop();
        return parts.join('/');
    },

    /**
     * Remove file extension
     * @param {string} filename - Filename with extension
     * @param {string} ext - Extension to remove
     * @returns {string} Filename without extension
     */
    trimExtension: function(filename, ext) {
        if (filename.toLowerCase().endsWith(ext.toLowerCase())) {
            return filename.slice(0, -ext.length);
        }
        return filename;
    },
    
    /**
     * Format frames to timecode (HH:MM:SS:FF)
     * @param {number} frames - Duration in frames
     * @param {number} fps - Frames per second (default: 24)
     * @returns {string} Formatted timecode
     */
    formatTimecode: function(frames, fps = 24) {
        const totalSeconds = frames / fps;
        const h = Math.floor(totalSeconds / 3600);
        const m = Math.floor((totalSeconds % 3600) / 60);
        const s = Math.floor(totalSeconds % 60);
        const f = Math.floor(frames % fps);
        
        return h.toString().padStart(2, '0') + ':' + 
               m.toString().padStart(2, '0') + ':' + 
               s.toString().padStart(2, '0') + ':' + 
               f.toString().padStart(2, '0');
    },
    
    /**
     * Parse timecode back to frames (for comparison)
     * @param {string} timecode - Timecode string (HH:MM:SS:FF)
     * @param {number} fps - Frames per second (default: 24)
     * @returns {number} Total frames
     */
    parseTimecode: function(timecode, fps = 24) {
        if (!timecode || timecode === '') return 0;
        
        const parts = timecode.split(':');
        if (parts.length !== 4) return 0;
        
        const h = parseInt(parts[0]) || 0;
        const m = parseInt(parts[1]) || 0;
        const s = parseInt(parts[2]) || 0;
        const f = parseInt(parts[3]) || 0;
        
        return (h * 3600 + m * 60 + s) * fps + f;
    },
    
    /**
     * Process an OTIO file and create an ExtJS store from it
     * @param {File} file - The OTIO file from input or drop event
     * @returns {Promise<Ext.data.Store>} Promise resolving to the media store
     */
    processOTIOFile: function(file) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            
            reader.onload = (event) => {
                try {
                    const content = event.target.result;
                    const result = this.parseOTIO(content);
                    
                    if (!result.success) {
                        reject(new Error(result.error));
                        return;
                    }
                    
                    // Create and return a store with the media items
                    const mediaStore = Ext.create('Ext.data.Store', {
                        fields: [
                            {name: 'type', type: 'string'},
                            {name: 'fileName', type: 'string'},
                            {name: 'path', type: 'string'},
                            {name: 'originalFile', type: 'string'},
                            {name: 'displayName', type: 'string'},
                            {name: 'usedDurationTC', type: 'string'},
                            {name: 'availableDurationTC', type: 'string'},
                            // Image sequence specific fields
                            {name: 'baseUrl', type: 'string'},
                            {name: 'namePrefix', type: 'string'},
                            {name: 'nameSuffix', type: 'string'},
                            {name: 'padding', type: 'int'},
                            {name: 'fullStartFrame', type: 'int'},
                            {name: 'fullEndFrame', type: 'int'},
                            {name: 'frameCount', type: 'int'},
                            {name: 'previewPath', type: 'string'},
                            // Container path for special types
                            {name: 'containerPath', type: 'string'}
                        ],
                        data: result.mediaItems
                    });
                    
                    resolve(mediaStore);
                } catch (error) {
                    reject(error);
                }
            };
            
            reader.onerror = (error) => {
                reject(error);
            };
            
            reader.readAsText(file);
        });
    }
});