#!/bin/sh
#
# Example code how to backup an object to the P5 Object Archive
# using publicly available AWS CLI utility.
# For retrieving files, look in aws_restore.sh example script.
#
# Currently this example script is just a simple wrapper that
# does nothing else but maintain the same command like as it's
# restore companion and passing the arguments directly to the
# AWS CLI utility.
#
# It is recommended to create separate profile for AWS CLI for accessing
# files on the P5 Object Archive. Note: when creating profiles with the
# AWS configure utility, leave the region name and output format to "None".
#
# Usage:
#
#   sh aws_backup.sh URL Bucket_Name File_Path Destination_Path AWS_Profile
#
# Notes:
#
# URL is in form http://ip:port or https://ip:port
# Standard for P5 Object Archive: http port is 8000, https 8443.
#
# Bucket_Name is the name of the P5 Object Bucket.
#
# File_Path is the path of the file which is to be copied over
# to the P5 Object Archive server.
#
# Destination_Path is name of the object to be stored on the
# P5 Object Archive. The name may contain '/' separators
# but it MUST NOT begin with one and MUST NOT be a single '.'
#
# AWS_Profile is optional, will use "default" AWS profile, if ommited.
#
# An example of how to backup an object to the P5 Object Archive:
#
#   sh aws_backup.sh http://192.168.1.200:8000 MyBucket MyFile.zip MyObject p5-s3
#
# See the file "license.txt" for information on usage and
# redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
#############################################################################

#
# Check usage and identify parameters
#

if test $# -le 3; then
    echo "Usage: url bucket path destination profile=default"
    exit 1
fi

url=$1
bucket=$2
path=$3
destination=$4

if test $# -eq 5; then
    profile=$5
else
    profile=default
fi

#
# Simply pass arguments down to the AWS CLI
#

aws --profile $profile --endpoint-url $url s3 cp $path s3://$bucket/$destination

exit 0
