/**
 * MainController.js - Controller for Dashboard Events
 * 
 * Handles all user interactions and state changes in the dashboard.
 * 
 * Key methods:
 * @method onColumnRadioChange - Handles column layout changes
 *   - Only processes if oldValue exists (prevents initial render triggering)
 *   - Clears dashboard state
 *   - Sends new column count to server
 *   - Triggers page reload via markToolRow
 * 
 * @method onWidgetCheckChange - Handles widget visibility changes
 *   - Ensures widget array format for single/multiple selections
 *   - Clears state and updates server configuration
 * 
 * @method onDashboardReset - Resets dashboard to default state
 *   - Clears client state and server configuration
 */

Ext.define('Dashboard.view.MainController', {
    extend: 'Ext.app.ViewController',
    alias: 'controller.dashboard',
    
    init: function() {
        var me = this;
        var view = me.getView();
        var state = Ext.state.Manager.get(view.stateId);
        
        if (!state || !state.items || state.items.length === 0) {
            Ext.state.Manager.clear(view.stateId);
        }

        // Add widget height management
        this.initWidgetHeightManagement();

    },

    initWidgetHeightManagement: function() {
        var me = this;
        var view = me.getView();
        var dashboard = view.down('dashboard');
        const widgetHeightsKey = 'p5-dashboard' + '-widget-heights';

        // HERE BEGINS THE NEW WIDGET HEIGHT MANAGEMENT
        // The state management implemented by extjs overwrites the saved height when loading the page.
        // Apply widget heights after complete rendering
        dashboard.on('afterrender', function() {
            try {
                var storedHeights = JSON.parse(localStorage.getItem(widgetHeightsKey) || '{}');
                //console.log('Retrieved stored heights:', storedHeights);
                
                dashboard.items.each(function(column) {
                    if (column && column.items && column.items.each) {
                        column.items.each(function(widget) {
                            var widgetId = widget.itemId || widget.id;
                            if (widgetId && storedHeights[widgetId]) {
                                //console.log('Setting height for widget:', widgetId, storedHeights[widgetId]);
                                widget.setHeight(storedHeights[widgetId]);
                            }
                        });
                    }
                });
            } catch(e) {
                console.error('Error applying widget heights:', e);
            }
        });
        
        // Save widget heights on resize
        dashboard.on('resize', function() {
            if (!dashboard.rendered || !dashboard.el) return;
            
            try {
                var heights = {};
                var hasChanges = false;
                
                dashboard.items.each(function(column) {
                    if (column && column.items && column.items.each) {
                        column.items.each(function(widget) {
                            var widgetId = widget.itemId || widget.id;
                            if (widgetId) {
                                var height = widget.getHeight();
                                if (height > 0) {
                                    heights[widgetId] = height;
                                    hasChanges = true;
                                }
                            }
                        });
                    }
                });
                
                if (hasChanges) {
                    localStorage.setItem(widgetHeightsKey, JSON.stringify(heights));
                    //console.log('Saved widget heights after resize:', heights);
                }
            } catch(e) {
                console.error('Error saving widget heights after resize:', e);
            }
        });
    },

    onColumnRadioChange: function(radiogroup, newValue, oldValue) {
        if (Object.keys(oldValue).length > 0) {
            var dashboard = this.getView().down('dashboard');
            var newColumnCount = newValue.columns;

            // State analysis to determine current widgets
            var state = Ext.state.Manager.get(dashboard.stateId);
            var activeWidgets = state ? state.items.map(function(w) { return w.type; }) : [];

            Ext.state.Manager.clear(dashboard.stateId);
    
            Ext.Ajax.request({
                url: document.dashboardForm.startTdp.value,
                params: {
                    caller: 'ajaxWidgetSet',
                    columnCount: newColumnCount,
                    widgets: Ext.JSON.encode(activeWidgets),
                    widgetPositions: ''
                },
                success: function(response) {
                    markToolRow('home:0', 'GettingStarted');
                }
            });
        }
    },

    onWidgetCheckChange: function(menuItem, newValue, oldValue) {
        if (Object.keys(oldValue).length > 0) {
            var dashboard = this.getView().down('dashboard');
            var activeWidgets = Array.isArray(newValue.widgets) ? newValue.widgets : [newValue.widgets];
            
            // State analysis for number of columns
            var state = Ext.state.Manager.get(dashboard.stateId);
            var columnCount = state && state.columnWidths ? state.columnWidths.length : 1;
            
            // Widget-Positions-Mapping erstellen
            var widgetPositions = {};
            if (state && state.items) {
                state.items.forEach(function(item) {
                    if (activeWidgets.indexOf(item.type) !== -1) {
                        widgetPositions[item.type] = item.columnIndex;
                    }
                });
            }
            // Spalten-Belegung zählen
            var widgetsInColumn = new Array(columnCount).fill(0);
            for (var type in widgetPositions) {
                if (widgetPositions.hasOwnProperty(type) && activeWidgets.indexOf(type) !== -1) {
                    var colIndex = widgetPositions[type];
                    if (colIndex < columnCount) {
                        widgetsInColumn[colIndex]++;
                    }
                }
            }
            
            // Neue Widgets in die Spalten mit weniger Inhalt einfügen
            activeWidgets.forEach(function(type) {
                if (!widgetPositions.hasOwnProperty(type)) {
                    // Neue Widgets bekommen die Spalte mit den wenigsten Widgets
                    var colIndex = 0;
                    for (var i = 1; i < columnCount; i++) {
                        if (widgetsInColumn[i] < widgetsInColumn[colIndex]) {
                            colIndex = i;
                        }
                    }
                    widgetPositions[type] = colIndex;
                    widgetsInColumn[colIndex]++;
                }
            });
            
            // State löschen
            Ext.state.Manager.clear(dashboard.stateId);
            
            Ext.Ajax.request({
                url: document.dashboardForm.startTdp.value,
                params: {
                    caller: 'ajaxWidgetSet',
                    columnCount: columnCount,
                    widgets: Ext.JSON.encode(activeWidgets),
                    widgetPositions: Ext.JSON.encode(widgetPositions)
                },
                success: function(response) {
                    markToolRow('home:0', 'GettingStarted');
                }
            });
        }
    },
    
    onDashboardReset: function() {
        var dashboard = this.getView().down('dashboard');

        Ext.state.Manager.clear(dashboard.stateId);

        Ext.Ajax.request({
            url: document.dashboardForm.startTdp.value,
            params: {
                caller: 'ajaxDashboardReset',
            },
            success: function(response) {
                markToolRow('home:0', 'GettingStarted');
            }
        });
    },

    onWidgetRefresh: function(event, toolEl, owner) {
        var panel = owner.ownerCt;
        // Suche nach Grids und Cartesian Charts
        var components = panel.query('grid, cartesian, dataview');
        
        // only stores with the url /lexxapp are reloaded
        components.forEach(function(component) {
            var store = component.getStore();
            if (store && store.getProxy()) {
                var url = store.getProxy().getUrl();
                if (url && url.indexOf('/lexxapp') === 0) {
                    store.reload();
                }
            }
        });
    },

    onNotificationClick: function() {
        var dashboard = this.getView().down('dashboard');
        var state = Ext.state.Manager.get(dashboard.stateId);
        var activeWidgets = state ? state.items.map(function(w) { return w.type; }) : [];
        
        var notificationIndex = activeWidgets.indexOf('notification');
        if (notificationIndex !== -1) {
            activeWidgets.splice(notificationIndex, 1);
        } else {
            activeWidgets.push('notification');
        }
        
        Ext.state.Manager.clear(dashboard.stateId);
        
        Ext.Ajax.request({
            url: document.dashboardForm.startTdp.value,
            params: {
                caller: 'ajaxWidgetSet',
                widgets: Ext.JSON.encode(activeWidgets),
                widgetPositions: ''
            },
            success: function(response) {
                markToolRow('home:0', 'GettingStarted');
            }
        });

    },

    onInstallInfoLoad: function(store) {
        var record = store.findRecord('param', 'notifications');
        if (record && record.get('value') > 0 ) {
            this.getViewModel().set('hasNotifications', true);
        }
    },

    onWidgetExport: function(event, toolEl, owner) {
        var panel = owner.ownerCt;
        var widget = panel.down('grid, cartesian, dataview');
        var widgetTitle = panel.title || 'data';
        
        if (widget.isXType('grid')) {
            // Für Grid-Widgets direkter Export der angezeigten Daten
            this.exportGridData(widget, widgetTitle);
        } else if (widget.isXType('cartesian')) {
            // Für Chart-Widgets - Export des zugrundeliegenden Stores
            this.exportChartData(widget, widgetTitle);
        } else if (widget.isXType('dataview')) {
            // Für Dataview-Widgets
            this.exportDataViewData(widget, widgetTitle);
        }
    },
    
    // Export von Grid-Daten (mit Spaltenüberschriften und Formatierung)
    exportGridData: function(grid, widgetTitle) {
        var data = [];
        var columns = grid.getColumns();
        var store = grid.getStore();
        
        // Header-Zeile aus sichtbaren Spalten erstellen
        var headers = [];
        columns.forEach(function(column) {
            if (!column.hidden) {
                headers.push(column.text || column.dataIndex);
            }
        });
        data.push(headers.join('\t'));
        
        // Daten-Zeilen erzeugen
        store.each(function(record) {
            var row = [];
            columns.forEach(function(column) {
                if (!column.hidden) {
                    var value;
                    
                    // exportValue-Attribut prüfen, falls vorhanden
                    if (column.exportValue === 'raw') {
                        // Rohdaten aus dem Store verwenden
                        value = record.get(column.dataIndex);
                    } 
                    else if (typeof column.exportValue === 'function') {
                        // Benutzerdefinierte Export-Formatierungsfunktion verwenden
                        value = column.exportValue(record.get(column.dataIndex), record);
                    }
                    else {
                        // Standard: Rohdaten aus dem Store
                        value = record.get(column.dataIndex);
                    }
                    
                    // Datum formatieren, falls nötig
                    if (value instanceof Date) {
                        value = Ext.Date.format(value, 'Y-m-d H:i:s');
                    }
                    
                    row.push(value !== undefined && value !== null ? value : '');
                }
            });
            data.push(row.join('\t'));
        });
        
        // TSV exportieren
        this.downloadTSV(data, widgetTitle);
    },
     
    // Export von Chart-Daten
    exportChartData: function(chart, widgetTitle) {
        var store = chart.getStore();
        var data = [];
        
        if (store) {
            // Header-Zeile aus allen Fields im Store erstellen
            var fields = store.model.getFields();
            var headers = [];
            
            fields.forEach(function(field) {
                // Bestimmte Felder überspringen
                if (field.name === 'dateShort' || 
                    field.name === 'dataAmountGiB' || 
                    field.name === 'id') {
                    return; // continue für forEach
                }
                headers.push(field.name);
            });
            
            data.push(headers.join('\t'));
            
            // Daten-Zeilen
            store.each(function(record) {
                var row = [];
                
                fields.forEach(function(field) {
                    // Bestimmte Felder überspringen
                    if (field.name === 'dateShort' || 
                        field.name === 'dataAmountGiB' || 
                        field.name === 'id') {
                        return; // continue für forEach
                    }
                    
                    var value = record.get(field.name);
                    if (field.type === 'date' && value instanceof Date) {
                        value = Ext.Date.format(value, 'Y-m-d H:i:s');
                    }
                    row.push(value !== undefined && value !== null ? value : '');
                });
                
                data.push(row.join('\t'));
            });
        }
        
        // TSV exportieren
        this.downloadTSV(data, widgetTitle);
    },

    // Export von DataView-Daten
    exportDataViewData: function(dataview, widgetTitle) {
        var store = dataview.getStore();
        var data = [];
        
        if (store) {
            // Header-Zeile aus allen Fields im Store erstellen
            var fields = store.model.getFields();
            var headers = [];
            
            fields.forEach(function(field) {
                // Bestimmte Felder überspringen
                if (field.name === 'dateShort' || 
                    field.name === 'dataAmountGiB' || 
                    field.name === 'id') {
                    return; // continue für forEach
                }
                headers.push(field.name);
            });
            
            data.push(headers.join('\t'));
            
            // Daten-Zeilen
            store.each(function(record) {
                var row = [];
                
                fields.forEach(function(field) {
                    // Bestimmte Felder überspringen
                    if (field.name === 'dateShort' || 
                        field.name === 'dataAmountGiB' || 
                        field.name === 'id') {
                        return; // continue für forEach
                    }
                    
                    var value = record.get(field.name);
                    if (field.type === 'date' && value instanceof Date) {
                        value = Ext.Date.format(value, 'Y-m-d H:i:s');
                    }
                    row.push(value !== undefined && value !== null ? value : '');
                });
                
                data.push(row.join('\t'));
            });
        }
        
        // TSV exportieren
        this.downloadTSV(data, widgetTitle);
    },
    
    // Gemeinsame Download-Funktion
    downloadTSV: function(data, widgetTitle) {
        // TSV-String erstellen
        var tsvContent = data.join('\n');
        
        // Download mit Zeitstempel
        var now = new Date();
        var dateStr = Ext.Date.format(now, 'Y-m-d-H-i');
        var blob = new Blob([tsvContent], { type: 'text/tab-separated-values' });
        var link = document.createElement('a');
        link.href = window.URL.createObjectURL(blob);
        link.download = widgetTitle.replace(/\s+/g, '_').toLowerCase() + '-' + dateStr + '.tsv';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

});