##############################################################################
#
# main_lib.tcl
#
#    P5 main TCL Library
#
#    See the file "license.txt" for information on usage and
#    redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
#    Rcsid: @(#)$Id: main_lib.tcl,v 1.295 2019/03/14 11:00:53 zv Exp $
#
##############################################################################

proc fmtdatetime { value { fmt "" } } {

    if { $fmt == "" } {
        set fmt [session::get cacheUserSettingDateTime]
    }

    if { $value == 0 || $value == "" } {
        return ""
    }

    try {
        switch -exact -- $fmt {
            doNotFormat {
                return $value
            }
            formatDateInt {
                clock format $value -format "%d-%b-%Y"
            }
            formatDateTimeInt {
                clock format $value -format "%d-%b-%Y %H:%M"
            }
            formatDateTimeLongDe {
                clock format $value -format "%a %d.%b.%Y %H:%M"
            }
            formatDateTimeShortDe {
                clock format $value -format "%d.%b.%Y %H:%M"
            }
            formatDateTimeDe {
                clock format $value -format "%d.%m.%y %H:%M"
            }
            formatDateTimeLongUSA {
                clock format $value -format "%a %b %d, %Y %I:%M%p"
            }
            formatDateTimeShortUSA {
                clock format $value -format "%d %b %Y %I:%M%p"
            }
            formatDateShortUSA {
                clock format $value -format "%d %b %Y"
            }
            formatDateTimeUSA {
                clock format $value -format "%m/%d/%y %I:%M%p"
            }
            formatTimeShort {
                clock format $value -format "%d-%b %H:%M"
            }
            formatDateTimeJa {
                clock format $value -format "%y/%m/%d %I:%M%p"
            }
            formatDateTimeISOShort {
                clock format $value -format "%Y-%m-%d"
            }
            formatDateTimeISOLong {
                clock format $value -format "%Y-%m-%dT%H:%M"
            }
            default {
                return $value
            }
        }
    } catch {*} {
        ns_log warning "fmtdatetime: '$value' invalid time"
        return $value
    } trymsg
}

##############################################################################

proc resolveDate { kval } {

    try {
        set now [clock seconds]
        set cDay [clock format $now -format %d]
        set cMon [clock format $now -format %m]
        set cYea [clock format $now -format %y]
        # given day -> expand with month and year
        # given day and month -> expand with year
        # given day, month and year
        if { \
        [regexp {^(=|<|>|<=|>=)?(\d{1,2})$} $kval match oper cDay] || \
        [regexp {^(=|<|>|<=|>=)?(\d{1,2})\.(\d{1,2})$} $kval match oper cDay cMon] || \
        [regexp {^(=|<|>|<=|>=)?(\d{1,2})\.(\d{1,2})\.(\d{1,2})$} $kval match oper cDay cMon cYea] } {
            if { $oper == "" || $oper == "=" } {
                # ns_log notice "select a given day"
                set sTime [clock scan "00:00:00 $cMon/$cDay/$cYea"]
                set eTime [clock scan "23:59:59 $cMon/$cDay/$cYea"]
            } elseif { $oper == "<" } {
                # older than a given date
                # ns_log notice "select younger than a given date"
                set sTime [clock scan "00:00:00 01/01/70"]
                set eTime [expr [clock scan "23:59:59 $cMon/$cDay/$cYea"] - 24 * 3600]
            } elseif { $oper == "<=" } {
                # older or equals than a given date
                # ns_log notice "select younger than a given date"
                set sTime [clock scan "00:00:00 01/01/70"]
                set eTime [clock scan "23:59:59 $cMon/$cDay/$cYea"]
            } elseif { $oper == ">" } {
                # younger than a given date
                # ns_log notice "select younger than a given date"
                set sTime [expr [clock scan "00:00:00 $cMon/$cDay/$cYea"] + 24 * 3600]
                set eTime $now
            } elseif { $oper == ">=" } {
                # younger or equals than a given date
                # ns_log notice "select younger than a given date"
                set sTime [clock scan "00:00:00 $cMon/$cDay/$cYea"]
                set eTime $now
            }
        # older than xx days
        } elseif { [regexp {^\+(\d+)$} $kval match bDay] } {
            # ns_log notice "select older than $bDay days"
            set sTime [clock scan "00:00:00 01/01/70"]
            set eTime [expr {$now - $bDay * 24 * 3600}]
        # younger than xx days
        } elseif { [regexp {^\-(\d+)$} $kval match aDay] } {
            # ns_log notice "select younger than $aDay days"
            set sTime [expr {$now - $aDay * 24 * 3600}]
            set eTime $now
        }
        return [list $sTime  $eTime]
    } catch {*} {
        utility::error "Invalid date format"
    } trymsg

}

##############################################################################

proc fmtduration {time} {

    # Calculates and rounds time, adds unit
    #
	return [utility::formatDuration $time 1]
}

##############################################################################

proc fmtfilesize { value { type "" } } {

    if { $type == "Directory" || $type == "Container" || $type == "Link" || $type == "Noaccess"  || $value == "" } {
        return "-"
    }

    return [utility::calc_size $value 1]

}

##############################################################################

proc msgfmt { message { level "" } } {

    if { $level != "" } {
        if { $level == "errorinfo" } {
            ns_log error "[login::procpage]: $message $::errorInfo"
        } elseif { $level == "error" } {
            ns_log error "[login::procpage]: $message"
        } elseif { $level == "warning" } {
            ns_log warning "[login::procpage]: $message"
        } else {
            ns_log notice "[login::procpage]: $message"
        }
    }
    #
    return [translate $message]

}

##############################################################################

proc fmtmsg { story message { level "" } } {

    set message [translate $message]
    set bold ""

    if { $level == "errorinfo" } {
        ns_log error "[login::procpage]: $message $::errorInfo"
        set img "/lexxapp/img4/error_16.png"
        set bold "color: #333333;font-weight: bold"
    } elseif { $level == "error" } {
        ns_log error "[login::procpage]: $message"
        set img "/lexxapp/img4/error_16.png"
        set bold "color: #333333;font-weight: bold"
    } elseif { $level == "warning" } {
        ns_log warning "[login::procpage]: $message"
        set img "/lexxapp/img4/warning_16.png"
    } elseif { $level == "notice" } {
        ns_log notice "[login::procpage]: $message"
        set img "/lexxapp/img4/notice_16.png"
    } elseif { $level == "okay" } {
        set img "/lexxapp/img4/check-12.png"
    } else {
        set img "/lexxapp/img4/step_16.png"
    }

    set story "$story\n<table><tr><td style='width:20px;padding-left:5px'><img src=$img></td><td style='padding-left:8px;$bold'>$message</td></tr></table>"

    return $story

}

##############################################################################

proc fmtmsg2 { story part1 part2 { level "" } } {

    if { $level != ""   } {
        ns_log ${level} "$part1 -> $part2"
        set level "[translate $level]: "
    }
    #
    set part1 [translate $part1]
    if { $story == "" } {
        return "$level$part1: $part2"
    } else {
        return "$story\n$level$part1: $part2"
    }

}

##############################################################################

proc fmtalert { story message { level "" } } {


    if { $level == "errorinfo" } {
        ns_log error "[login::procpage]: $message $::errorInfo"
        set level "[translate "Error"]: "
    } elseif { $level == "error" } {
        ns_log error "[login::procpage]: $message"
        set level "[translate "Error"]: "
    } elseif { $level == "warning" } {
        ns_log warning "[login::procpage]: $message"
        set level "[translate "Warning"]: "
    } elseif { $level == "notice" } {
        ns_log notice "[login::procpage]: $message"
        set level "[translate "Note"]: "
    }

    #
    if { $story == "" } {
        return "$level$message"
    } else {
        return "$story\n$level$message"
    }

}

##############################################################################

proc getKeyValuePairsForUpdate { qset } {

    set attribute     ""
    set keyValuePairs ""
    #
    set allAttributesOnPage [ns_set get $qset allAttributesOnPage]
    foreach singleAttributeOnPage $allAttributesOnPage {
        set pageAttribute($singleAttributeOnPage) [ns_set get $qset $singleAttributeOnPage]
    }
    #
    set pageAttributes [array names pageAttribute]
    set dbAttributes [session::get dbAttributes]
    foreach attribute $pageAttributes {
        if { [lsearch -exact $dbAttributes $attribute] == -1 } {
            continue
        }
        if { $pageAttribute($attribute) != [session::get dbAttribute($attribute)] } {
            # ns_log notice "Attribute update request from USER \"[session::user]\" ELEMENT \"$pageAttribute(name)\" ATTRIBUTE \"$attribute\" OLD=\"[session::get dbAttribute($attribute)]\" NEW=\"$pageAttribute($attribute)\""
            lappend keyValuePairs $attribute $pageAttribute($attribute)
        }
    }
    return $keyValuePairs
}

##############################################################################

proc getDateTime {} {

    return [clock seconds]

}

##############################################################################

proc getOutputFormat { upArray } {

    upvar $upArray localArray

    foreach attribute [keylkeys cacheAttributes etc.attributes] {
        keylget cacheAttributes etc.attributes.$attribute.name name
        if { ![keylget cacheAttributes etc.attributes.$attribute.ooformat localArray($name)] } {
            set localArray($name) "doNotFormat"
        }
    }
    return 1

}

##############################################################################

proc translate { word { id 0 } } {

    # Workaround during developping
    # return $word

    set lang [session::browserLanguage]
    set rtc  runtimeconfig
    set supp etc.languages.supported

    #
    # Do not attempt translations for built-in or
    # unsupported languages.
    #

    if {$lang == "en"
        ||
        [sv_exists $rtc $supp] == 0
        ||
        [sv_lsearch $rtc $supp $lang] == -1} {
        return $word
    }

    set lankey etc.languages.$lang

    #
    # Look for id cached translations first.
    #
    if { $id > 0 && [sv_get $rtc $lankey.$id trans] } {
        if {$trans != ""} {
            return $trans
        }
    }

    #
    # Look for sting cached translations second.
    #
    if { [sv_get $rtc $lankey.$word trans] } {
        if {$trans == ""} {
            #ns_log notice "No translation for: $word"
            return $word
        } else {
            return $trans
        }
    }

    #
    # Try to translate strings with more than 2 words only.
    # We thus try to match all available dynamic strings
    # and if we get a match, translate and cache it.
    #

    set dynkey etc.languages.dynstr.$lang

    if {[llength [split $word { }]] > 2} {
        foreach key [sv_array names $rtc etc.languages.dynexp.*] {
            lassign [sv_get $rtc $key] regstr regvars
            if {[eval regexp [list $regstr] [list $word] _ $regvars]} {
                set theid [lindex [split $key "."] end]
                set mystr [sv_get $rtc $dynkey.$theid]
                if {$mystr == ""} {
                    break ; # No given translation
                }
                if {[catch {subst -nocomm $mystr} trans]} {
                    #ns_log notice "No translation for: $word"
                    return $word
                }
                return [sv_set $rtc $lankey.$word $trans]
            }
        }
    }

    # cache also not translated words

    sv_set $rtc $lankey.$word ""

    # ns_log notice "No translation for: $word"
    return $word

}

##############################################################################

proc charset {} {

    set lang [session::browserLanguage]

    switch -exact -- $lang {
        ja      { return "Shift_JIS" }
        default { return "utf-8" }
    }
}

##############################################################################

proc getLang {} {

    set lang [session::browserLanguage]

    switch -exact -- $lang {
        de {
            return de
        }
        fr {
            return fr
        }
        ja {
            return ja
        }
        default {
            return en
        }
    }
}

##############################################################################

proc loadLanguages { languagesDB } {

    ns_log notice "Reading translations from: '$languagesDB'"

    set rsrcgroup [TranslationGroup new -volatile]

    set langlist  ""
    set rtc       runtimeconfig
    set rtcroot   etc.languages

    set langlist [const GUIlanguages]

    #
    # Cache supported languages
    #

    set local($rtcroot.supported) $langlist

    #
    # Loop over all defined translation records and
    # cache their system strings and ID's
    #

    $rsrcgroup foreach my {

        #
        # Caches for string and ID search
        #

        set myStr  [$my describe]
        set myID   [$my name]
        set myKeys [$my read]

        foreach lang $langlist {
            set trans [lkeyget $myKeys $lang]
            # create string cache
            set local($rtcroot.$lang.$myStr) $trans
            # create id cache
            set local($rtcroot.$lang.$myID)  $trans

            #
            # This is still not used; disable to save space
            #

            if {0} {
                foreach app [lkeyget $myKeys appList] {
                    set local($rtroot.$lang.$app.$myStr) $trans
                }
            }
        }

        #
        # Build the regexp expression to match the original message.
        # This will yield, for the given myStr (that is, the source
        # of the translated string):
        #
        #   "P5 client $client with release $release connected"
        #
        # ... the "regstr" like this:
        #
        #   "P5 client ([^ ]+) with release ([^ ]+) connected"
        #

        set str [string map {( \\( ) \\) [ \\[ ] \\]} $myStr]
        if {![regsub -all {\$[^'\" ]+} $str {([^'\" ]+)} regstr]} {
            continue ; # Has no variables
        }

        #
        # Build the list of variables located in the myStr. This
        # list is then combined with the regular expression above
        # to match the passed string in the "translate" procedure.
        # For the above example, the "regvars" vill contain:
        #
        #   client release
        #

        set regvars ""

        foreach var [lassign [regexp -inline $regstr $myStr] whole] {
            lappend regvars [string trimleft $var {$}]
        }

        # Cache both regexp/regvars list and the source string
        # for all available languages.
        foreach lang $langlist {
            set local($rtcroot.dynstr.$lang.$myID) [lkeyget $myKeys $lang]
        }
        set local($rtcroot.dynexp.$myID) [list $regstr $regvars]
    }

    sv_array set $rtc [array get local]

    ns_log notice "Reading translations: done"
}

##############################################################################

proc mklist { jslist } {

    set result ""

    foreach el $jslist {
        if { $el eq {} } continue
        lappend result $el
    }
    if {[llength $result] == 1 && [lindex $result 0] eq {}} {
        return
    }

    return $result
}

##############################################################################

proc mkfilelist { jslist } {

    set result ""

    foreach se $jslist {
        if {$se ne {}} {
            lappend result [mkfilepath $se]
        }
    }

    return $result
}

##############################################################################

proc mkfilelist2 { jslist } {

    set result ""

    foreach se $jslist {
        if {$se ne {}} {
            lappend result $se
        }
    }

    return $result
}

##############################################################################

proc mkJson2List { json } {

    set result ""

    if {[catch {set el [utility::json2dict $json] }] || $el == "" } { set el $json }

    foreach str $el {
        lappend result [subst -nobackslashes -nocommands -novariables $str]
    }

    return $result

}

##############################################################################

proc mkfilepath { path } {

    set path [utility::file normalize $path]

    # we disabled this converting because it causes problems on Windows and
    # it is according to Zoran not required any longer for all other
    # plattforms

    # set path [macchar::normalize nfd $path]

    return $path
}

##############################################################################

proc userInGroup { groups } {
    # check user in group
    set user [session::user]
    if {[session::isadmin]} {
        return 1
    }
    foreach group $groups {
        if { [utility::isInGroup $user $group] == 1 } {
            return 1
        }
    }
    return 0
}

##############################################################################

proc loadCustomerConfig { { configDB customerconfig } } {

    ns_log dev "loadCustomerConfig: Update customer configuration cache."

    ####### loginareas
    try {
        set user         [session::user]
        ns_log dev       "loadCustomerConfig: caching login areas for user \"$user\""
        set loginareagrp [LoginAreaGroup new -volatile]
        set databaseagrp [DatabaseGroup new -volatile]
        set clientgrp    [ClientGroup new -volatile]
        set licensegrp   [LicenseGroup new -volatile]
        set usrprefgrp   [UserPrefGroup new -volatile]
        set servergrp    [ServerGroup new -volatile]
        set loginareas   [$loginareagrp ls]
        set databases    [$databaseagrp ls]
        set clients      [$clientgrp ls]
        set servers      [$servergrp ls]
        set workstgrp    [WorkstationGroup new -volatile]
        set syncgrp      [SyncPlanGroup new -volatile]
        #
        # create a configuration view depended form the activated licenses
        #
        set resourcesToView [list licenses]
        set isLicensed      0
        if { [$licensegrp lictypefree ArchivePlan] != 0 } {
            lappend resourcesToView archiveplans devices filters pools volumes var databases loginareas users
            set isLicensed 1
        }
        if { [$licensegrp lictypefree BackupPlan] != 0 } {
            lappend resourcesToView backupplans devices filters pools volumes var databases loginareas users
            set isLicensed 1
        }
        if { [$licensegrp lictypefree SyncPlan] != 0 } {
            lappend resourcesToView syncplans filters var databases loginareas users
            set isLicensed 1
        }
        if { [$licensegrp lictypefree Backup2Go] != 0 } {
            lappend resourcesToView backup2go workstations var users filters
            set isLicensed 1
        }
        if { [$licensegrp lictypefree Jukebox] != 0 } {
            lappend resourcesToView jukeboxes
        }
        if { [$licensegrp lictypefree MigrationPlan] != 0 } {
            lappend resourcesToView migrationplans
        }
        if { $servers != "" && $resourcesToView == "licenses" } {
            set resourcesToView [list servers filters]
        }
        lappend resourcesToView licenses clients
        set resourcesToView [lsort -unique $resourcesToView]
        session::put resourcesToView $resourcesToView
        #
        # create a list of all loginareas accessable for this user
        #
        keylset lareas category [list]
        foreach loginarea $loginareas {
            set access 1
            set url     ""
            set iDriver ""
            set usage   "unknown index"
            set status  "Enabled"
            set client  _dummy_
            set hfsla   ""
            set cl		""
            set thisRes [$loginareagrp $loginarea]
            set text    [$thisRes describe]
            set path    [$thisRes loginPath]
            set groups  [$thisRes grouplist]
            set tree    [$thisRes tree]
            if { [$thisRes status] == "Disabled" } continue
            if { $groups != "" && ![userInGroup $groups] } continue
            if { [set category [$thisRes category]] == "" } { set category $tree }
            if { [lsearch -exact $databases $tree] == -1 } {
                set status "Database '$tree' is not in configuration"
            } else {
                set thisIndex [$databaseagrp $tree]
                set iDriver   [$thisIndex driver]
                set usage     [$thisIndex usage]
                set iVersion  [$thisIndex vers]
                if { ($iDriver == "AsxTree"  && [$licensegrp lictypefree ArchivePlan] == 0)  || ($iDriver == "BsxTree"  && [$licensegrp lictypefree BackupPlan] == 0) || ($iDriver == "HfsTree" && $isLicensed == 0 ) } {
                    continue
                } elseif { [$thisIndex status] == "Disabled" } {
                    set status "Database '$tree' is disabled"
                } elseif { [set client [$thisIndex client]] != "" && [lsearch -exact $clients $client] != -1  && [[$clientgrp $client] status] == "Disabled" } {
                    set status "Client '$client' is disabled"
                } else {
                    set url [encodeLexxUrl $tree $path]
                }
            }
            # for file picker
            if { $iDriver == "AsxTree" || $iDriver == "BsxTree" || $iDriver == "HfsTree"  || $iDriver == "WstTree" } {
                keylset lareas $iDriver.$category $loginarea
            }
            # for loginarea browser
            # to get a list of categories sorted by usage
            keylset lareas $usage.$category $loginarea
            # to get a list of loginarea sorted by user
            keylset lareas database.$user.$tree $path
            #
            keylset lareas category.$category.$loginarea.client $client
            keylset lareas category.$category.$loginarea.status $status
            #
            # remove description form text
            if { [regexp {(Backup index for client) (.*)} $text match rest cl] } {
                set text $cl
            } elseif { [regexp {(Filesystem from client) (.*)} $text match rest cl] } {
                set text  $cl
                set hfsla $cl
            }
            keylset lareas category.$category.$loginarea.text    $text
            keylset lareas category.$category.$loginarea.hfsla   $hfsla
            keylset lareas category.$category.$loginarea.path    $path
            keylset lareas category.$category.$loginarea.url     $url
            keylset lareas category.$category.$loginarea.usage   $usage
            keylset lareas category.$category.$loginarea.iDriver $iDriver
            keylset lareas category.$category.$loginarea.index   $tree
            # for start page
            keylset lareas loginarea.$loginarea.status    $status
            keylset lareas loginarea.$loginarea.url       $url
            keylset lareas loginarea.$loginarea.usage     $usage
            keylset lareas loginarea.$loginarea.category  $category
            keylset lareas loginarea.$loginarea.text      $text
        }
        # for workstation user
        # download not open with first release 3.1
        # released with 4.3
        set category "Workstation Data Repository"
        if { [session::isadmin] } {
            set wstlist [$workstgrp ls]
        } else {
            set wstlist [UserPref $user workstationList]
        }
        foreach myWst $wstlist {
            # skip all templates < 10000
            if { [$workstgrp ls $myWst] != "" && $myWst > 10000 } {
                set thisWst [$workstgrp $myWst]
                set db "wst_localhost"
                set tx [$thisWst describe]
                set st [$thisWst status]
                if { $st == "Disabled" } {
                    set st "Workstation $tx is disabled"
                }
                set ur [encodeLexxUrl $db "/"]
                keylset lareas category.$category.$myWst.client $myWst
                keylset lareas category.$category.$myWst.status $st
                keylset lareas category.$category.$myWst.text   $tx
                keylset lareas category.$category.$myWst.path   "/"
                keylset lareas category.$category.$myWst.url    $ur
                keylset lareas category.$category.$myWst.usage  Workstation
                keylset lareas category.$category.$myWst.iDriver ""
                keylset lareas Workstation.$category $tx
            } else {
                ns_log error "could not read login area information for user '$user' from workstation '$myWst'"
            }
        }
        # to start manual sync from target location of sync plan
        # released with 5.4
        set loginarea "_dummy_"
        set category  "Storage location of sync plans"
        $syncgrp foreach thisPlan {
            if { ![$thisPlan enabled] } { continue }
            set loginarea  "SyncPlan_[$thisPlan name]"
            set targetHost [$thisPlan targetHost]
            set targetDir  [$thisPlan targetDir]
            set text       [$thisPlan describe]
            set url        "hfs_$targetHost#$targetDir"
            #
            # skip if user has no access to the tree
            if { [catch { set path [keylget lareas database.$user.hfs_${targetHost}] } err] } { continue }
            #
            keylset lareas category.$category.$loginarea.text    "$text ($targetHost:$targetDir)"
            keylset lareas category.$category.$loginarea.path    $targetDir
            keylset lareas category.$category.$loginarea.url     $url
            keylset lareas category.$category.$loginarea.iDriver "HfsTree"
            keylset lareas category.$category.$loginarea.status  "Enabled"
            keylset lareas category.$category.$loginarea.usage   Synchronize
            keylset lareas category.$category.$loginarea.client  $targetHost
            keylset lareas Synchronize.$category $loginarea
        }
        #
        #
        keylset lareas lareasEpoch  [$loginareagrp epoch]
        keylset lareas clientEpoch  [$clientgrp epoch]
        keylset lareas licenseEpoch [$licensegrp epoch]
        keylset lareas usrprefEpoch [$usrprefgrp epoch]
        #
        session::put loginareas $lareas
        ns_log dev "loadCustomerConfig: caching login areas done."
    } catch {*} {
        ns_log error "could not read login areas: $trymsg $::errorInfo"
    } trymsg

    return 1
}

##############################################################################

proc loadUserConfig { user } {

    # ns_log notice "loadUserConfig: Reading preferences for user \"$user\"."

    try {

        set resourcegrp  [UserPrefGroup new -volatile]
        set keyList      [UserPrefResource keys]
        set duser        $user
        if { [$resourcegrp ls $duser] == "" } {
            ns_log warning "no settings found for user '$user' using generic default"
            set thisResource [$resourcegrp generic]
        } else {
            set thisResource [$resourcegrp $duser]
        }
        foreach key $keyList {
            keylset userSettings etc.user.$user.$key [$thisResource $key]
        }
        session::put rtc.userpref.$user $userSettings
        # ns_log notice "loadUserConfig: completed."
    } catch {*} {
       ns_log error "could not get user settings: $::errorInfo"
    } trymsg

    return 1
}

##############################################################################

proc operatorCanDo { operation } {

    set cacheUserOperations  [session::get cacheUserOperations]

    # check if operation is allowed for a user and for a specific database
    if { [lsearch $cacheUserOperations $operation] != -1 } {
        return 1
    } else {
        return 0
    }

}

##############################################################################

proc userCanDo { operation } {

    set cacheUserOperations  [session::get cacheUserOperations]
    set cacheIndexOperations [session::get cacheIndexOperations]

    # check if operation is allowed for a user and for a specific database
    if { [session::isadmin] || ([lsearch $cacheUserOperations $operation] != -1 && [lsearch $cacheIndexOperations $operation] != -1) } {
        return 1
    } else {
        return 0
    }

}

##############################################################################

proc getAttributeS { hd attribute } {

    #
    #
    try {
        return [$::lexxCommand getAttribute $hd $attribute]
    } catch {LEXXTREE NO_PERM} {
        return [translate "No permission to access element."]
    }
    #
    #
}

##############################################################################

proc getAttributeM { hd upArray attributeSet { resourcegrp {} }} {

    # ns_log notice "Time in getAttributeM proc: [time {    }]"
    upvar $upArray elementArray
    #
    # preload array
    array set elementArray [list name "" text "" type "" kind "" mtime "" btime "" elementurl "" database "" icon "" volid "" size "" cliptype  "" clippath "" iconpath "" clipaddr "" container ""]
    #
    set elementArray(database)         [$::lexxCommand set database]
    #set elementPath                    [$::lexxCommand elementPath $hd]
    #set elementArray(elementPath)      [encodeLexxUrl $elementArray(database) $elementPath]
    set elementArray(elementurl)       [encodeLexxUrl $elementArray(database) [$::lexxCommand elementAddress $hd]]
    #set elementArray(parentPath)       [file dirname $elementPath]

    #
    try {
        $::lexxCommand getAttribute1 $hd elementArray
    } catch {LEXXTREE BAD_HANDLE} {
        set elementArray(name)   [format {%s (%s)} '$hd' [translate "Inaccessible element"]]
        set elementArray(type)   "File"
    } catch {LEXXTREE NO_PERM} {
        # no_perm removed in 5.3 (next possible time also removed in GUI)
        set elementArray(name)   [translate "Permission denied"]
        set elementArray(type)   "File"
    } catch {*} {
        # For debugging in the event we didn't catch all possible errors
        ns_log error getAttribute1 returned $msg
    } msg

    # path is occasionally not delivered by getAttribute1
    # and in the event of an error,
    # so we define it unconditionally with what we know to be
    # the correct path
    #set elementArray(path) $elementPath

    #
    set type [$::lexxCommand typeName $elementArray(type)]
    set elementArray(type)  $type
    set elementArray(icon)  $type
    #
    if { $elementArray(type) == "Directory" } {
        set elementArray(leaf) ""
        set elementArray(leafIcon) "leafIcon_Dir"
    } else {
        set elementArray(leaf) "true"
        set elementArray(leafIcon) "leafIcon_File"
    }
    #
    if { $elementArray(cliptype) != "" } {
        set pageroot [ns_server pagedir]
        set addr     [$::lexxCommand getClip $hd]
        set icon     [$::lexxCommand getIcon $hd]
        set clipaddr [$::lexxCommand set webhome]/$addr
        set iconaddr [$::lexxCommand set webhome]/$icon
        # for images guestype will return "image"
		# continue only if image type is not unknown
        if { [set elementArray(media) [lindex [split [ns_guesstype $pageroot/$clipaddr] /] 0]] != "*" } {
            if { $icon != "" } {
    			set elementArray(iconpath) [$::lexxCommand set webhome]/[utility::urlencode $icon]
            }
			set elementArray(clippath) [$::lexxCommand set webhome]/[utility::urlencode $addr]
			set elementArray(container) [utility::file extension $addr]
			#
			if { [info exists elementArray(previewSize)] && $elementArray(previewSize) ne "" && $elementArray(previewSize) > 64 } {
				set picBoxW  [expr $elementArray(previewSize) - 4  ]
			} else {
				set elementArray(previewSize) 160
				set picBoxW  156
			}
			set picBoxH  $picBoxW
			# set size to default
			set elementArray(xsize)  $picBoxW
			set elementArray(ysize)  $picBoxH
			# scale preview if larger the msize
			set msize $picBoxW
			if {![catch {ns_imgsize $pageroot/$iconaddr} isize]} {
				set oxsize [lindex $isize 0]
				set oysize [lindex $isize 1]
				set elementArray(xsize)  $oxsize
				set elementArray(ysize)  $oysize
				if { $oxsize > $msize || $oysize > $msize } {
					if { $oxsize >= $oysize } {
						set elementArray(ysize) [expr int($oysize / ( double($oxsize) / $msize )) ]
						set elementArray(xsize) $msize
					} else {
						set elementArray(xsize) [expr int($oxsize / ( double($oysize) / $msize )) ]
						set elementArray(ysize) $msize
					}
				}
			}
			# set type to Image
		    set elementArray(icon) "Image"
        }
    }
    #
    return 1

}

##############################################################################

proc getAttributeListing { hd upArray attributeSet { resourcegrp {} }} {

    # ns_log notice "Time in getAttributeM proc: [time {    }]"
    upvar $upArray elementArray
    #
    # preload array
    array set elementArray [list name "" text "" type "" kind "" mtime "" btime "" elementurl "" database "" icon "" volid "" size "" cliptype  "" clippath "" iconpath "" clipaddr "" container ""]
    #
    set elementArray(database)         [$::lexxCommand set database]
    set elementArray(elementurl)       [encodeLexxUrl $elementArray(database) [$::lexxCommand elementAddress $hd]]

    #
    try {
        $::lexxCommand getAttribute1 $hd elementArray
    } catch {LEXXTREE BAD_HANDLE} {
        set elementArray(name)   [format {%s (%s)} [file tail $elementPath] [translate "Inaccessible element"]]
        set elementArray(type)   "File"
    } catch {LEXXTREE NO_PERM} {
        # no_perm removed in 5.3 (next possible time also removed in GUI)
        set elementArray(name)   [translate "Permission denied"]
        set elementArray(type)   "File"
    } catch {*} {
        # For debugging in the event we didn't catch all possible errors
        ns_log error getAttribute1 returned $msg
    } msg

    #
    set type [$::lexxCommand typeName $elementArray(type)]
    set elementArray(type)  $type
    set elementArray(icon)  $type
    #
    if { $elementArray(type) == "Directory" } {
        set elementArray(leaf) ""
        set elementArray(leafIcon) "leafIcon_Dir"
    } else {
        set elementArray(leaf) "true"
        set elementArray(leafIcon) "leafIcon_File"
    }
    #
    #
    return 1

}

##############################################################################

proc getAttributeP4 { hd upArray attributeSet { resourcegrp {} }} {

    # ns_log notice "Time in getAttributeM proc: [time {    }]"
    upvar $upArray elementArray
    #
    # preload array
    array set elementArray [list name "" text "" type "" kind "" mtime "" btime "" elementurl "" database "" icon "" volid "" size "" cliptype  "" clippath "" clipaddr "" access ""]
    #
    set elementArray(database)         [$::lexxCommand set database]
    set elementPath                    [$::lexxCommand elementPath $hd]
    set elementArray(elementPath)      [encodeLexxUrl $elementArray(database) $elementPath]
    set elementArray(elementurl)       [encodeLexxUrl $elementArray(database) [$::lexxCommand elementAddress $hd]]
    set elementArray(parentPathByName) [file dirname $elementPath]
    #
    try {
        $::lexxCommand getAttribute1 $hd elementArray
        set elementArray(access) 1
    } catch {LEXXTREE BAD_HANDLE} {
        set elementArray(access) 1
        set elementArray(name)   [format {%s (%s)} [file tail $elementPath] [translate "Inaccessible element"]]
        set elementArray(type)   99
    } catch {LEXXTREE NO_PERM} {
        set elementArray(access) 0
        set elementArray(name)   [$::lexxCommand getAttribute $hd name]
        set elementArray(type)   [$::lexxCommand getAttribute $hd type]
        set elementArray(kind)   [$::lexxCommand getAttribute $hd kind]
    }
    #
    set elementArray(type)  [$::lexxCommand typeName $elementArray(type)]
    set elementArray(kind)  [$::lexxCommand kindName $elementArray(kind)]
    set elementArray(icon)  [getIconForType $elementArray(type) $elementArray(kind)]
    #
    # ns_log notice  elementArray:[array get elementArray]
    #
    return 1

}

##############################################################################

proc restoreCallback { numfiles totalsize { rmessage "" } } {

    if { [session::get collectForRestoreSignal] == "stop" || [session::get collectForRestoreTimeout] < [clock seconds] } {
        return -code 42 "Restore selection terminated by user"
    }
    set keylistRestore [session::get keylistRestore]
    keylset keylistRestore totalsize $totalsize
    keylset keylistRestore numfiles  $numfiles
    keylset keylistRestore rmessage  $rmessage
    session::put keylistRestore $keylistRestore
}

##############################################################################

proc collectForRestore { client index fileliste nosnapshot retainpath } {

    set filesTrigger   100
    set callback       lexxapp::restoreCallback

    try {
        set tree [mountDatabase $index [session::user] [session::password]]
        if { $nosnapshot eq 1 } {
            ns_log notice "Folder backup time is used as reference time used for restore selection."
        } else {
            ns_log notice "Index snapshot time is used as reference time used for restore selection."
        }
        #
        set vollist [$tree config -uvl]
        if {[llength $vollist] > 0} {
            ns_log notice "Volume(s) used for restore: $vollist"
        }
        #
        set hd [$tree restoreSelection -nosnapshot $nosnapshot -retainpath $retainpath $client $fileliste $callback $filesTrigger]
        #
        set sel [ResSel new $hd]
        set keylistRestore [session::get keylistRestore]
        keylset keylistRestore selectHandle $hd
        keylset keylistRestore volumes      [$sel volumes]
        keylset keylistRestore stores       [$sel stores]
        session::put keylistRestore $keylistRestore
    } catch {*} {
        ns_log error "collectForRestore: $::errorInfo"
        session::put collectForRestoreError $trymsg
        return -1
    } finally {
        catch { $sel destroy }
    } trymsg
    #
    return 1
}

##############################################################################

proc restorePlus { client index fileliste relocatePath { vtime 0 }} {

	set rtimeFlag 0

    try {
    	set dbgrp    [DatabaseGroup new]
        set tree     [$dbgrp mount $index]
		set treeType [$tree treetype]

		if { $treeType == "AsxTree" } { set rtimeFlag 1 }

        if { $vtime > 0 } {
            $tree config -vtime $vtime
        }

        if { $relocatePath != "" } {
            set rhd   [$tree restoreSelection -retainpath 0 $client $fileliste]
        } else {
            set rhd   [$tree restoreSelection -retainpath 1 $client $fileliste]
        }

        set sel   [ResSel new $rhd]
        set vols  [$sel volumes]
        set stor  [$sel stores]

        ns_log notice "Volume(s)      : $vols"
        ns_log notice "Cloud Store(s) : $stor"
        ns_log notice "Restore client : $client"
        ns_log notice "Conflict res.  : rename"

        if { $relocatePath != "" } {
        	JobMgr restore $client $rhd -relocatepath $relocatePath -resolution "rename" -setrtime $rtimeFlag -database $index
        } else {
        	JobMgr restore $client $rhd                             -resolution "rename" -setrtime $rtimeFlag -database $index
        }

    } catch {*} {
        ns_log error "restorePlus: $::errorInfo"
    } finally {
        catch { $dbgrp destroy }
    } trymsg

}

##############################################################################

proc migrationPlus { index pool fileliste targetIndex relocatePath } {

    try {
    	set dbgrp    [DatabaseGroup new]
        set tree     [$dbgrp mount $index]
		set treeType [$tree treetype]

        $tree config -vtime 0

        if { $relocatePath != "" } {
            set rhd   [$tree restoreSelection -retainpath 0 "localhost" $fileliste]
        } else {
            set rhd   [$tree restoreSelection -retainpath 1 "localhost" $fileliste]
        }

        set sel   [ResSel new $rhd]
        set vols  [$sel volumes]
        set stor  [$sel stores]

        ns_log notice "Restore selec. : $rhd"
        ns_log notice "Volume(s)      : $vols"
        ns_log notice "Cloud Store(s) : $stor"
        ns_log notice "Target pool    : $pool"
        ns_log notice "Target path    : $relocatePath"

        if { $relocatePath != "" } {
        	JobMgr migration -tgtpool $pool -selection $rhd -xroot $relocatePath -indexdb $targetIndex
        } else {
        	JobMgr migration -tgtpool $pool -selection $rhd -indexdb $targetIndex
        }

    } catch {*} {
        ns_log error "migrationPlus: $::errorInfo"
    } finally {
        catch { $dbgrp destroy }
    } trymsg

}

##############################################################################

proc irecoverCallback { } {

    if { [session::get collectForIRecoverSignal] == "stop" || [session::get collectForIRecoverTimeout] < [clock seconds] } {
        return -code 42 "Restore selection terminated by user"
    }
}

##############################################################################

proc migrationCallback { numfiles totalsize { rmessage "" } } {

    if { [session::get collectForMigrationSignal] == "stop" || [session::get collectForMigrationTimeout] < [clock seconds] } {
        return -code 42 "Migration selection terminated by user"
    }
    set keylistMigration [session::get keylistMigration]
    keylset keylistMigration totalsize $totalsize
    keylset keylistMigration numfiles  $numfiles
    keylset keylistMigration rmessage  $rmessage
    session::put keylistMigration $keylistMigration
}

##############################################################################

proc collectForMigration { planId } {

    set filesTrigger   10000
    set callback       lexxapp::migrationCallback

    try {
        set migdict [JobMgr migration_volumes $planId $callback $filesTrigger]
        #
        set keylistMigration [session::get keylistMigration]
        keylset keylistMigration volumes [dict get $migdict volumes]
        keylset keylistMigration numfiles [dict get $migdict numfiles]
        keylset keylistMigration totalsize [dict get $migdict numkbytes]
        session::put keylistMigration $keylistMigration
    } catch {*} {
        ns_log error "collectForMigration: $::errorInfo"
        session::put collectForMigrationError $trymsg
        return -1
    } finally {
        catch { $sel destroy }
    } trymsg
    #
    return 1
}

##############################################################################

proc checkIndex { index } {

    set callback lexxapp::irecoverCallback

    try {
        set resgrp [DatabaseGroup new -volatile]
        set thisDb   [$resgrp $index]
        set indexTyp [$thisDb driver]
        if { $indexTyp == "BsxTree" && [$thisDb check] == 0 } {
            set ok [$thisDb recover hd]
            if {$ok} {
                if {$hd ne {}} {
                    set sel [ResSel new $hd]
                    set keylistIRecover [session::get keylistIRecover]
                    keylset keylistIRecover selectHandle $hd
                    keylset keylistIRecover volumes      [$sel volumes]
                    session::put keylistIRecover $keylistIRecover
                } else {
                    keylset keylistIRecover selectHandle ""
                    keylset keylistIRecover volumes      ""
                    session::put keylistIRecover $keylistIRecover
                }
                return 1
            } else {
                return 0
            }
        } else {
            return 0
        }
    } catch {*} {
        ns_log error "checkIndex: $::errorInfo"
        session::put collectForIRecoverError $trymsg
        return -1
    } finally {
        catch { $sel   destroy }
    } trymsg
}

##############################################################################

proc collectForDownload { selElements } {

    variable klOfFiles ""
    variable homepath  ""
    try {
        set klOfFiles [session::get klOfFiles]
        foreach elementurl $selElements {
            set elementPath [decodeLexxUrl elementpath $elementurl]
            set database    [decodeLexxUrl database $elementurl]
            set dbhandle    [mountDatabase $database [session::user] [session::password]]
            set elemethd    [$dbhandle elementHandle $elementPath]
            set elementPath [$dbhandle elementPath $elemethd]
            set homepath    [file dirname $elementPath]
            set result      [loopForDownload $dbhandle $elemethd]
        }
    } catch {*} {
        ns_log error "collectForDownload: $::errorInfo"
        session::put collectForDownloadError $trymsg
        return -1
    } trymsg
    #
    session::put klOfFiles $klOfFiles
    return $result
}

##############################################################################

proc loopForDownload { dbhandle elemethd} {

    variable homepath
    variable klOfFiles
    set result 1
    #
    try {
        if { [session::get collectForDownloadSignal] == "stop" || [session::get collectForDownloadTimeout] < [clock seconds] } {
            return 0
        }
        if { [$dbhandle typeName [$dbhandle getAttribute $elemethd type]] == "Directory" } {
            keylset klOfFiles numdirs [expr [keylget klOfFiles numdirs] + 1]
            foreach childid [$dbhandle elementChildren $elemethd] {
                set result [loopForDownload $dbhandle $childid]
            }
        } else {
            set elementPath   [$dbhandle elementPath $elemethd]
            set elementName   [file tail $elementPath]
            set elementParent [file tail [file dirname $elementPath]]
            keylset klOfFiles totalsize [expr {[keylget klOfFiles totalsize] + [$dbhandle getAttribute $elemethd size]}]
            # ns_log notice totalsize:[keylget klOfFiles totalsize]
            set relpath [string range $elementPath [string length $homepath] end]
            if {[string index $relpath 0] == "/"} {
                set relpath [string range $relpath 1 end]
            }
            set filenr [keylget klOfFiles numfiles]
            keylset klOfFiles files.$filenr.abspath $elemethd
            keylset klOfFiles files.$filenr.relpath $relpath
            keylset klOfFiles numfiles [incr filenr]
        }
    } catch {FILTER} {
        return
    } catch {*} {
        ns_log error "loopForDownload: $::errorInfo"
        session::put collectForDownloadError $trymsg
        return -1
    } trymsg
    #
    return $result
}

##############################################################################

proc collectDownload { dbhandle elemethd { include {} } } {

    variable listDownFiles
    variable pathDownFiles
    set result 1
    #
    try {

        if { [$dbhandle typeName [$dbhandle getAttribute $elemethd type]] == "Directory" } {
            keylset listDownFiles numdirs [expr [keylget listDownFiles numdirs] + 1]
            foreach childid [$dbhandle elementChildren $elemethd] {
                set result [collectDownload $dbhandle $childid $include]
            }
        } else {
            set elementPath   [$dbhandle elementPath $elemethd]
            set elementName   [file tail $elementPath]
            set elementParent [file tail [file dirname $elementPath]]
            if { ![regexp $include $elementName] } {
                throw FILTER
            }
            keylset listDownFiles totalsize [expr [keylget listDownFiles totalsize] + [$dbhandle getAttribute $elemethd size]]
            set relpath [string range $elementPath [string length $pathDownFiles] end]
            if {[string index $relpath 0] == "/"} {
                set relpath [string range $relpath 1 end]
            }
            set filenr [keylget listDownFiles numfiles]
            keylset listDownFiles files.$filenr.abspath $elemethd
            keylset listDownFiles files.$filenr.relpath $relpath
            keylset listDownFiles numfiles [incr filenr]
        }
    } catch {FILTER} {
        return
    } catch {*} {
        ns_log error "collectDownload: $::errorInfo"
        return -1
    } trymsg
    #
    return $result
}

##############################################################################

proc getIconForType { type { kind "" } } {

    switch -exact -- $type {
        Directory - Container  {
            switch -exact -- $kind {
                Mountpoint  { return [session::url /img4/mount_point.png] }
                Link        { return [session::url /img4/ord_link.png] }
                default     { return [session::url /img4/ord.png] }
            }
        }
        File {
            switch -exact -- $kind {
                Link        { return [session::url /img4/unix_link.png]}
                Image       { return [session::url /img4/image.png]}
                default     { return [session::url /img4/unix.png]}
            }
        }
        License     { return [session::url /img4/license_16.png]}
        LoginArea   { return [session::url /img4/loginarea_16.png]}
        Client      { return [session::url /img4/clients_16.png]}
        Workstation { return [session::url /img4/workstation_16.png]}
        Server      { return [session::url /img4/server_16.png]}
        Backup2Go   { return [session::url /img4/workstations_16.png]}
        Translation { return [session::url /img4/translate_16.png]}
        Database    { return [session::url /img4/database_16.png]}
        UserPref    { return [session::url /img4/user_pref_16.png] }
        DeviceType  { return [session::url /img4/unknown.png] }
        JukeboxType { return [session::url /img4/unknown.png] }
        Device      { return [session::url /img4/TAPE_device_16.png]}
        Jukebox     { return [session::url /img4/jukebox_16.png]}
        Pool        { return [session::url /img4/pools_16.png]}
        ArchivePlan { return [session::url /img4/scheduler_16.png]}
        BackupPlan  { return [session::url /img4/scheduler_16.png]}
        SyncPlan    { return [session::url /img4/scheduler_16.png]}
        MigrationPlan { return [session::url /img4/scheduler_16.png]}
        Volume      { return [session::url /img4/volume_16.png]}
        Job         { return [session::url /img4/job_16.png]}
        Filter      { return [session::url /img4/filter_16.png] }
        Log         { return [session::url /img4/log_16.png]}
        Noaccess    { return [session::url /img4/no_access_16.png]}
        default     { return [session::url /img4/unknown.png]}
    }
}

##############################################################################

proc createNodeElement { hd elementtype { elementName "" } } {

    set elementtypeid [$::lexxCommand typeId $elementtype]
    if { $elementName != "" } {
        set hd [$::lexxCommand createNodeElement $hd $elementtypeid $elementName]
    } else {
        set hd [$::lexxCommand createNodeElement $hd $elementtypeid]
    }
    #
    return
}

##############################################################################

proc getSortCommand { column direction } {

    if { $column == "ctime" } {
        set command "lsort -integer    -index 1"
    } else {
        set command "lsort -dictionary -index 1"
    }
    if { $direction == "inc" } {
        set command "$command -increasing"
    } else {
        set command "$command -decreasing"
    }
    return $command
}

##############################################################################

proc pwdName { nextUrl } {

    set hd [$::lexxCommand elementHandle [decodeLexxUrl elementpath $nextUrl]]

    set name [$::lexxCommand getAttribute $hd name]

    if { [$::lexxCommand set database] == "customerconfig" } {
        return [translate $name]
    } else {
        return $name
    }
}

##############################################################################

proc getTreeStepPath { nextUrl } {

    set pwdPath [decodeLexxUrl elementpath $nextUrl]
    set pwdHd   [$::lexxCommand elementHandle $pwdPath]

    ns_log notice pwdHd:$pwdHd


    return
}

##############################################################################

proc download {  } {

    #
    # We must be logged-in correctly. Also, if somebody tampers
    # with the url, we invalidate session immediately.
    #

    if {[login::valid] == 0} {
        ns_log warning "not in session while downloading: [ns_conn url]"
        return [ns_returnforbidden]
    }

    #
    # Full-path of file to be downloaded is appended to url and urlencoded.
    # The url must have been in the form "/lexxapp/download/...." so we
    # start after first "download" string.
    #

    set urllist [ns_conn urlv]
    set index [lsearch $urllist "download"]

    if {$index == -1} {
        ns_log warning "malformed download url: [ns_conn url]"
        login::logout1
        return [ns_returnnotfound]
    }

    set pathList [lrange [ns_conn urlv] [incr index] end]

    if {[llength $pathList] == 0} {
        ns_log notice "download file path missing from url"
        login::logout1
        return [ns_returnnotfound]
    }

    set osPath [ns_normalizepath [ns_urldecode [eval file join $pathList]]]

    #
    # Allow only files from valid workspace (loginarea) to be downloaded.
    #

    set workspace [lindex [session::get treeStepName] 0]
    if { ![regexp "^$workspace" $osPath] } {
        ns_log warning "file $osPath is not part of login area $workspace"
        login::logout1
        return [ns_returnforbidden]
    }

    try {

        #
        # See if file there; skip if directory
        #

        if {![file exists $osPath]} {
            ns_log warning "download file not found: $osPath"
            return [ns_returnnotfound]

        } elseif {[file type $osPath] == "directory"} {
            ns_log warning "download file points to a directory: $osPath"
            login::logout1
            return [ns_returnforbidden]
        }

        #
        # See if we're allowed to read the file.
        # FIXME: portability problem for Windows !?
        #

        array set perms [file attributes $osPath]

        if {[session::user] == $perms(-owner)} {
            set mask 0400
        } elseif {[utility::isInGroup [session::user] $perms(-group)]} {
            set mask 0040
        } else {
            set mask 0004
        }

        if {[expr {$perms(-permissions) & $mask}] == 0} {
            return [ns_returnforbidden]
        }

        #
        # Returns file as MacBinary II stream if "mb" found in query.
        # Just existence of "mb" is tested, the value is ignored.
        # Otherwise return the plain file as octet stream.
        #

        set qs [ns_conn form]

        if {[string length $qs] && [ns_set ifind $qs "mb"] >= 0} {
            download::macbinary $osPath
        } else {
            download::binary $osPath
        }

    } catch {*} {

        #
        # Dang !
        #
        # Error happened in above try block. Invalidate session !
        #

        ns_log error "could not download $osPath: $::errorInfo"
        login::logout1

    } trystat
}

##############################################################################

proc splitElementLists { elements } {

    #
    set newlist [list]
    set elements [split $elements "\r\n"]
    foreach element $elements {
        if { $element == "" } continue
        # set newlist [concat $newlist [list $element]]
        lappend newlist $element
    }
    return $newlist
    #
}

##############################################################################

proc checkResourceName { name } {

    if { [regexp "^_|_$" $name match rest] == 1 } {
        error "resources begin or end with '_' are reserved for system use"
    }
    if { [regexp {[Ll]exx} $name match rest] == 1 } {
        error "'$match' as part of a resource name is reserved for system use"
    }
    if { [regexp {^[A-Za-z0-9\_-]+$} $name match rest] == 0 } {
        error "Invalid character in resource name. Valid are 'A-Z', 'a-z', '0-9', '-', '_'"
    }
}

##############################################################################

proc encodeLexxUrl { database elementpath } {

    # FIXED with format command
    return "$database#$elementpath"
    #
}

##############################################################################

proc decodeLexxUrl { key lexxurl } {

    if { $lexxurl == "" || ![regexp {([^#]+)#(.*)} $lexxurl match database elementpath] } {
        return
    }
    #
    switch -exact -- $key {
        "database" {
            return $database
        }
        "elementpath" {
            return $elementpath
        }
        default {
            return [list $database $elementpath]
        }
    }
}

##############################################################################

proc mountDatabase { database user {pass ""} } {

    #
    # ns_log notice "Time in mountDatabase proc, step switch: [time {    }]"
    #
    #try {
    #} catch {*} {
    #    utility::error "mountDatabase: could not mount database: '$database': $::errorInfo"
    #} trymsg

        set snapshot ""
        set thisDb  [$::dbgrp $database]
        set dbtype  [$thisDb driver]
        #
        if { $dbtype == "HfsTree" } {
            set cmd [$::dbgrp mount $database $user $pass]
            $cmd config -mod 0777
            $cmd set options(access) rw
        } elseif { $dbtype == "DbsTree" } {
            set cmd [$::dbgrp mount $database $user $pass]
            $cmd config -mod 0xFFF
            $cmd set options(access) rw
        } elseif { $dbtype == "AsxTree" } {
            set cmd [$::dbgrp mount $database $user $pass]
            $cmd config -mod 0xFFF
            $cmd set options(access) rw
            if { [set indexoption [session::get $database.indexoption]] == "" } {
                keylset indexoption setvolume ""
                keylset indexoption setstores ""
                keylset indexoption setvtime  0
                session::put $database.indexoption $indexoption
            }
            $cmd config -uvl       [keylget indexoption setvolume]
            $cmd config -uol       [keylget indexoption setstores]
            $cmd config -vtime     [keylget indexoption setvtime]
        } elseif { $dbtype == "WstTree" } {
        	set workstation [session::get workstation]
            set cmd [$::dbgrp mounton $database $workstation $user $pass]
            # mount to bypase authentication (a dark hole from zoran)
            $cmd mount $workstation
            $cmd config -mod 0777
            $cmd set options(access) rw
            if { [set indexoption [session::get $database.indexoption]] == "" } {
                keylset indexoption snapshot ""
                session::put $database.indexoption $indexoption
            }
            set snapshot [keylget indexoption snapshot]
            $cmd config -snapshot $snapshot
        } elseif { $dbtype == "BsxTree" } {
            set cmd    [$::dbgrp mount $database $user $pass]
            $cmd set options(access) r
            if { [set indexoption [session::get $database.indexoption]] == "" } {
                keylset indexoption setvtime  0
                keylset indexoption setvolume ""
                keylset indexoption setstores ""
                keylset indexoption setpool   ""
                session::put $database.indexoption $indexoption
            }
            $cmd config -vtime [keylget indexoption setvtime]
            $cmd config -uvl   [keylget indexoption setvolume]
            $cmd config -uol   [keylget indexoption setstores]
        } else {
            set cmd    [$::dbgrp mount $database $user $pass]
        }
        # set internal database attribute
        $cmd set database $database
        $cmd set snapshot $snapshot
        $cmd set webhome  [$thisDb webpath]
        # from 5.6 use session variable
        # $cmd set dbusage  [$thisDb usage]
    #
    return $cmd
    #
}

##############################################################################

proc trunc { word nr } {

    if { [string length $word] > $nr  } {
        return "[string range $word 0 [expr $nr - 3]]..."
    } else {
        return $word
    }

}

##############################################################################

proc getJobStatus { state returnCode } {

    switch -exact -- $state {
        scheduled {
            return [translate "Scheduled but unexpectedly interrupted"]
        }
        started   {
            return [translate "Started but unexpectedly interrupted"]
        }
        cancelled   {
            return [translate "Cancelled"]
        }
        stopped   {
            return [translate "Stopped"]
        }
        terminated   {
            return [translate "Terminated"]
        }
        completed   {
            if       { $returnCode == [TCL_OK]      } {
                return [translate "Finished"]
            } elseif { $returnCode == [TCL_BREAK]   } {
                return [translate "Cancelled"]
            } elseif { $returnCode == [TCL_ERROR]   } {
                return [translate "Error"]
            } elseif { $returnCode == [TCL_WARNING] } {
                return [translate "Warning"]
            } else   {
                return [translate "Exception"]
            }
        }
        default   {
            return [translate "Unknown completed or interrupted"]
        }
    }
}

##############################################################################

proc cleanSet { qset } {

    set emptyKeys [list]
    for {set nr 0} { $nr < [ns_set size $qset] } { incr nr } {
        set keyname [ns_set key $qset $nr]
        if { [ns_set get $qset $keyname] == "" } {
            lappend emptyKeys $keyname
        }
    }
    foreach keyname $emptyKeys {
        ns_set delkey $qset $keyname
    }

}

##############################################################################

proc setPageCache { forPage toCache } {

    try {
        binary scan [md5c $forPage] H* page
        binary scan [md5c $toCache] H* md5CacheNew
        if { [session::get $page] == $md5CacheNew } return
        # store cache in session
        session::put $page $md5CacheNew
        set resourcegrp   [UserPrefGroup new -volatile]
        set duser         [session::user]
        set thisResource  [$resourcegrp $duser]
        # read cache
        set pageCache     [$thisResource pageCache]
        # set cache
        keylset pageCache $page $toCache
        # store cache
        $thisResource pageCache $pageCache
    } catch {*} {
        ns_log error "setPageCache: could not set page cache: $::errorInfo"
    } trymsg

}

##############################################################################

proc getPageCache { forPage { keys "" } } {
    try {
        set returnCache ""
        binary scan [md5c $forPage] H* page
        set resourcegrp  [UserPrefGroup new -volatile]
        set duser        [session::user]
        set thisResource [$resourcegrp $duser]
        set totalCache   [$thisResource pageCache]
        if { ![keylget totalCache $page pageCache] } { set pageCache "" }
        if { $keys != "" } {
			array set localArray $pageCache
			foreach {key default} $keys {
				if { [info exists localArray(${key})] } {
					lappend returnCache $key $localArray($key)
				} else {
					lappend returnCache $key $default
				}
			}
		} else {
			set returnCache $pageCache
		}
        return $returnCache
    } catch {*} {
        foreach key $keys {
            lappend returnCache $key ""
        }
        ns_log error "getPageCache: could not get page cache: $::errorInfo"
        return $returnCache
    }

}

##############################################################################

proc mkIntList { userInput } {

    #
    # This procedure returns a proper list of interger values
    # userInput: is a string of single numbers and ranges
    # e.g. "2-5, 8 9,10-20 25 8"
    # e.g of a result: "2 3 4 5 8 9 10 11 ..."
    #

    set returnList  ""

    #
    # First we check for valid character
    #
    if { [regexp {^[0-9 ,\-]+$} $userInput] } {
        #
        set workingList $userInput
        #
        # First step in analyzing user input is to get all ranges. Delimiter is "-"
        # The regexp is generating a list with three items: match sting and sub results
        #
        foreach { match start end } [regexp -inline -all {([0-9]+) *- *([0-9]+)} $userInput] {
            #
            # Create a list of integer values. (returnList)
            #
            if { $start <= $end } {
                for { set nr $start } { $nr <= $end } { incr nr } {
                    lappend returnList $nr
                }
            }
            #
            # remove this range from string
            #
            regsub -all $match $workingList {} workingList
        }
        #
        # Second step is to find all single integers. Delimiter are " " and ","
        # The regexp is generating a list with two items: match sting and result
        #
        foreach { match start } [regexp -inline -all {([0-9]+)[ ,]*} $workingList ] {
            lappend returnList $start
        }
        #
        # Sort the list of items an remove all duplicated entries.
        #
        set returnList [lsort -integer -unique $returnList]
        #
        #
        #
    }

    return $returnList

}


##############################################################################

proc downloadSupportFiles {{ path "" }} {

    set awhome [ns_info home]
    set config [const configDirectory]

    #
    # Logfiles from last 14 days.
    #

    set when [expr {[clock seconds] - (14 * 86400)}]

    foreach f [glob -nocomplain -dir [file join $awhome log] lexxsrv.*] {
        if {![catch {file mtime $f} mt] && $mt > $when} {
            lappend all $f
        }
    }

    #
    # Our own files
    #

    set ltfsdir [file join $awhome temp ltfsdir.txt]
    catch {
        file delete $ltfsdir
        set fd   [open $ltfsdir w]
        puts $fd [join [glob -tails -directory "$config/ltfsindex" */*] \n]
        close $fd
    }

    lappend all [glob -nocomplain -directory $config -type f {*lexxsrv.[0-9]*}]

    lappend all                  \
        "$awhome/libobjstore.log"  \
        "$config/customerconfig/resources.db" \
        "$config/customerconfig/resources.db-shm" \
        "$config/customerconfig/resources.db-wal" \
        $ltfsdir

    #
    # Platform specific files
    #

    switch -- $::tcl_platform(os) {
        SunOS {
            lappend all \
                "/var/adm/messages"
        }
        Darwin {
            set iof [file join $awhome temp ioreg.txt]
            catch {utility::sudo ioreg -l -w 0 > $iof}
            lappend all                            \
                "/Library/Logs/CrashReporter/nsd*" \
                "/var/log/system.log" $iof
        }
        Linux {
            lappend all             \
                "/proc/scsi/scsi"   \
                "/var/log/messages"
        }
    }
    # If Export is selected, all placeholders will be dissolved.
    if { $path ne "" } {
        set zip [file join $path P5_SupportData_[utility::machid].zip]
        foreach ff $all {
            foreach file [glob -nocomplain $ff] {
                lappend files $file
            }
        }
        utility::zip $zip $files
    } else {
        eval download::zip $all
    }
}

proc downloadAllSupportFiles {{ path "" }} {

    set awhome  [ns_info home]
    set config  [const configDirectory]

    #
    # Our own files
    #

    set ltfsdir [file join $awhome temp ltfsdir.txt]
    catch {
        file delete $ltfsdir
        set fd   [open $ltfsdir w]
        puts $fd [join [glob -tails -directory "$config/ltfsindex" */*] \n]
        close $fd
    }

    lappend all [glob -nocomplain -directory $config -type f {*lexxsrv.[0-9]*}]

    lappend all                  \
        "$awhome/libobjstore.log"  \
        "$awhome/log/lexxsrv.*"  \
        "$config/customerconfig/resources.db" \
        "$config/customerconfig/resources.db-shm" \
        "$config/customerconfig/resources.db-wal" \
        $ltfsdir

    #
    # Platform specific files
    #

    switch -- $::tcl_platform(os) {
        SunOS {
            lappend all \
                "/var/adm/messages*"
        }
        Darwin {
            set iof [file join $awhome temp ioreg.txt]
            catch {utility::sudo ioreg -l -w 0 > $iof}
            lappend all                            \
                "/Library/Logs/CrashReporter/nsd*" \
                "/var/log/system.log*" $iof
        }
        Linux {
            lappend all             \
                "/proc/scsi/scsi"   \
                "/var/log/messages" \
                "/var/log/boot.*"
        }
    }
    # If Export is selected, all placeholders will be dissolved.
    if { $path ne "" } {
        set zip [file join $path P5_SupportData_[utility::machid].zip]
        foreach ff $all {
            foreach file [glob -nocomplain $ff] {
                lappend files $file
            }
        }
        utility::zip $zip $files
    } else {
        eval download::zip $all
    }
}

##############################################################################

proc commonCalendarEvent { upArray eveObj } {

    upvar $upArray eve

    set id [$eveObj id]

    set weekList  [list 1 [translate "first" 11863] 2 [translate "second" 11864] 3 [translate "third" 11865] 4 [translate "fourth" 11866] 5 [translate "last" 11867]]
    set dayListL  [list 1 [translate "Monday"]  2 [translate "Tuesday"] 3 [translate "Wednesday"] 4 [translate "Thursday"] 5 [translate "Friday"] 6 [translate "Saturday"] 7 [translate "Sunday"] ]
    set dayListS  [list 1 [translate "Mon"]  2 [translate "Tue"] 3 [translate "Wed"] 4 [translate "Thu"] 5 [translate "Fri"] 6 [translate "Sat"] 7 [translate "Sun"] ]

    #
    # get all recurrences
    #
    set planRecur [$eveObj recurrence]
    if { $planRecur != "" } {
        set runWeek ""
        set runDays ""
        #
        # daily recurrence
        #
        if {[$planRecur isdaily]} {
            #
            # get freqenz
            #
            set frq [$planRecur count]
            if { $frq == "1" } {
                set eve($id,runFreq) [translate "Run daily"]
            } elseif { $frq > "1" } {
                set eve($id,runFreq) [translate "Run every $frq days"]
            } else {
                set eve($id,runFreq) "-"
            }
        #
        # weekly recurrence
        #
        } elseif {[$planRecur isweekly]} {
            #
            # get days the event will run
            #
            foreach {n d} $dayListS {
                if {[$planRecur isdayset $n]} {
                    lappend runDays $d
                }
            }
            set runDays [join [translate $runDays] ", "]
            #
            # get freqenz
            #
            set frq [$planRecur count]
            if { $frq == "1" } {
                set eve($id,runFreq) "[translate "Run weekly on"] $runDays"
            } elseif { $frq > "1" } {
                set eve($id,runFreq) "[translate "Run every $frq weeks on"] $runDays"
            } else {
                set eve($id,runFreq) "-"
            }
        #
        # monthly recurrence
        #
        } elseif {[$planRecur ismonthly]} {
            #
            # get week and day the event will run
            #
            set weekOfMonth [$planRecur week]
            set dayOfWeek   [$planRecur dayofweek]
            # get week of month
            foreach {n d} $weekList {
                if { $weekOfMonth == $n } {
                    set runWeek $d
                    break
               }
            }
            # get day of week
            foreach {n d} $dayListL {
                if { $dayOfWeek == $n } {
                    set runDay $d
                    break
               }
            }
            #
            # get freqenz
            #
            set frq [$planRecur count]
            if { $frq == "1" } {
                set eve($id,runFreq) "[translate "Run monthly on each"] [translate "$runWeek"] [translate "$runDay"]"
            } elseif { $frq > "1" } {
                set eve($id,runFreq) "[translate "Run every $frq months on each"] [translate "$runWeek"] [translate "$runDay"]"
            } else {
                set runFreq "-"
            }
        }
    }

    #
    # get all exceptions
    #

    set planSkip  [$eveObj exception]
    if { $planSkip != "" } {
        set runWeek ""
        set runDays ""
        #
        # daily recurrence
        #
        if {[$planSkip isdaily]} {
            #
            # get freqenz
            #
            set frq [$planSkip count]
            if { $frq == "1" } {
                set eve($id,skipFreq) [translate "except every day"]
            } elseif { $frq > "1" } {
                set eve($id,skipFreq) [translate "except every $frq days"]
            } else {
                set eve($id,skipFreq) "-"
            }
        #
        # weekly recurrence
        #
        } elseif {[$planSkip isweekly]} {
            #
            # get days the event will run
            #
            foreach {n d} $dayListS {
                if {[$planSkip isdayset $n]} {
                    lappend runDays $d
                }
            }
            set runDays [join [translate $runDays] ", "]
            #
            # get freqenz
            #
            set frq [$planSkip count]
            if { $frq == "1" } {
                set eve($id,skipFreq) "[translate "except every week on"] $runDays"
            } elseif { $frq > "1" } {
                set eve($id,skipFreq) "[translate "except every $frq weeks on"] $runDays"
            } else {
                set eve($id,skipFreq) "-"
            }
        #
        # monthly recurrence
        #
        } elseif {[$planSkip ismonthly]} {
            #
            # get week and day the event will run
            #
            set weekOfMonth [$planSkip week]
            set dayOfWeek   [$planSkip dayofweek]
            # get week of month
            foreach {n d} $weekList {
                if { $weekOfMonth == $n } {
                    set runWeek $d
                    break
               }
            }
            # get day of week
            foreach {n d} $dayListL {
                if { $dayOfWeek == $n } {
                    set runDay $d
                    break
               }
            }
            #
            # get freqenz
            #
            set frq [$planSkip count]
            if { $frq == "1" } {
                set eve($id,skipFreq) "[translate "except every month on each"] [translate "$runWeek"] [translate "$runDay"]"
            } elseif { $frq > "1" } {
                set eve($id,skipFreq) "[translate "except every $frq months on each"] [translate "$runWeek"] [translate "$runDay"]"
            } else {
                set eve($id,skipFreq) "-"
            }
        }

        set eve($id,runFreq) "$eve($id,runFreq)<br>$eve($id,skipFreq)"

    }
 }

##############################################################################

proc setCommonCalendarEvent { upArray eveObj } {

    upvar $upArray ca

    #
    # set recurrence
    #
    if { $ca(frequency) == "daily" } {
        set planRecur [$eveObj newrecurrence daily]

        #
        # set freqenz
        #

        $planRecur count $ca(dayFrq)
    #
    # weekly recurrence
    #
    } elseif { $ca(frequency) == "weekly" } {
        set planRecur  [$eveObj newrecurrence weekly]

        #
        # set freqenz and days the event will run
        #

        $planRecur count   $ca(weekFrq)
        $planRecur setdays $ca(isMon) $ca(isTue) $ca(isWed) $ca(isThu) $ca(isFri) $ca(isSat) $ca(isSun)

    #
    # monthly recurrence
    #
    } elseif { $ca(frequency) == "monthly" } {
        set planRecur [$eveObj newrecurrence monthly]

        #
        # set freqenz, week and day the event will run
        #

        $planRecur count     $ca(monthFrq)
        $planRecur week      $ca(weekOfMonth)
        $planRecur dayofweek $ca(dayOfWeek)
    #
    # none recurrence
    #
    } elseif { $ca(frequency) == "none" } {

        #
        # delete recurrence and exception
        #

        $eveObj delrecurrence
        $eveObj delexception
    }

    #
    # set days to skip Synchronize
    #

    if { $ca(skipfrequency) == "daily" } {
        set planSkip [$eveObj newexception daily]

        #
        # set freqenz
        #

        $planSkip count $ca(dayFrqSkip)
    #
    # weekly exception
    #
    } elseif { $ca(skipfrequency) == "weekly" } {
        set planSkip  [$eveObj newexception weekly]

        #
        # set freqenz and days the event will run
        #

        $planSkip count   $ca(weekFrqSkip)
        $planSkip setdays $ca(isMonSkip) $ca(isTueSkip) $ca(isWedSkip) $ca(isThuSkip) $ca(isFriSkip) $ca(isSatSkip) $ca(isSunSkip)

    #
    # monthly exception
    #
    } elseif { $ca(skipfrequency) == "monthly" } {
        set planSkip [$eveObj newexception monthly]

        #
        # set freqenz, week and day the event will run
        #

        $planSkip count     $ca(monthFrqSkip)
        $planSkip week      $ca(skipWeekOfMonth)
        $planSkip dayofweek $ca(skipDayOfWeek)
    #
    # none exception
    #
    } elseif { $ca(skipfrequency) == "none" } {

        #
        # delete exception
        #

        $eveObj delexception

    }
}

##############################################################################

proc getCommonCalendarEvent { upArray eveObj } {

    upvar $upArray ca

    set planRecur [$eveObj recurrence]
    set planSkip  [$eveObj exception]

    #
    # check for defined recurrence
    #
    if { $planRecur != "" } {
        #
        # daily recurrence
        #
        if {[$planRecur isdaily]} {
            set ca(frequency) daily
            #
            # get freqenz
            #
            set ca(dayFrq) [$planRecur count]
        #
        # weekly recurrence
        #
        } elseif {[$planRecur isweekly]} {
            set ca(frequency) weekly
            #
            # get freqenz and days the event will run
            #
            set ca(weekFrq) [$planRecur count]
            foreach {n d} [list 1 isMon 2 isTue 3 isWed 4 isThu 5 isFri 6 isSat 7 isSun] {
                if {[$planRecur isdayset $n]} {
                    set ca(${d}) $n
                }
            }
        #
        # monthly recurrence
        #
        } elseif {[$planRecur ismonthly]} {
            set ca(frequency) monthly
            #
            # get freqenz, week and day the event will run
            #
            set ca(monthFrq)    [$planRecur count]
            set ca(weekOfMonth) [$planRecur week]
            set ca(dayOfWeek)   [$planRecur dayofweek]
        }
    }
    #
    # check for defined exception
    #
    if { $planSkip != "" } {
        #
        # daily exception
        #
        if {[$planSkip isdaily]} {
            set ca(skipfrequency) daily
            #
            # get freqenz
            #
            set ca(dayFrqSkip) [$planSkip count]
        #
        # weekly exception
        #
        } elseif {[$planSkip isweekly]} {
            set ca(skipfrequency) weekly
            #
            # get freqenz and days the event will run
            #
            set ca(weekFrqSkip) [$planSkip count]
            foreach {n d} [list 1 isMonSkip 2 isTueSkip 3 isWedSkip 4 isThuSkip 5 isFriSkip 6 isSatSkip 7 isSunSkip] {
                if {[$planSkip isdayset $n]} {
                    set ca(${d}) $n
                }
            }
        #
        # monthly exception
        #
        } elseif {[$planSkip ismonthly]} {
            set ca(skipfrequency) monthly
            #
            # get freqenz, week and day the event will run
            #
            set ca(monthFrqSkip)    [$planSkip count]
            set ca(skipWeekOfMonth) [$planSkip week]
            set ca(skipDayOfWeek)   [$planSkip dayofweek]
        }
    }
}

##############################################################################

proc isInteger { var } {
	if { [string is integer -strict $var] && $var >= 0 } {
		return $var
	} else {
		return 0
	}
}

##############################################################################

proc isActive { nlayer clayer { existent "" } } {
	if { $existent != ""  } {
        if { $nlayer == $clayer } {
            return "is-active"
        } else {
            return "is-pasive"
        }
	} else {
		return "is-disable"
	}
}

##############################################################################

proc isActiveLic { nlayer clayer { existent "" } } {
	if { !$existent } {
        if { $nlayer == $clayer } {
            return "is-active"
        } else {
            return "is-pasive"
        }
	} else {
        if { $nlayer == $clayer } {
            return "is-active"
        } else {
            return "is-disable"
        }
	}
}

##############################################################################

proc createDiskLib { baseDir usageSwitch sizegb { poolname "" } } {


    ns_log notice baseDir:$baseDir
    ns_log notice usageSwitch:$usageSwitch
    ns_log notice sizegb:$sizegb
    ns_log notice poolname:$poolname

    set message              ""
    set driveprefix          "drive_"
    set drivetextprefix1     "Drive 1 of disk library "
    set drivetextprefix2     "Drive 2 of disk library "
    set jukeboxprefix        "disk_lib_"
    set jukeboxtextprefix    "Disk library "
    set backuppoolname       "Disk-Backup"
    set backuppooltext       "Default disk backup pool"
    set archivepoolname      "Disk-Archive"
    set archivepooltext      "Default disk archive pool"


    # set up the libary
    set slots                10
    set minvol               2
    set licmaxvol            [const maxDiskVolumeSize]
    # a requirement from the support to reduce this from 2048 to 1024
    set maxvol               1024
    set lowervollimit        [expr {wide(0x40000000) * $minvol}]
    set lowerborder          [expr {wide($lowervollimit) * $slots}]
    set uppervollimit        [expr {wide(0x40000000) * $maxvol}]
    set upperborder          [expr {wide($uppervollimit) * $slots}]
    #
    try {
        #
        set devicegrp  [DeviceGroup new -volatile]
        set jukeboxgrp [JukeboxGroup new -volatile]
        set poolgrp    [PoolGroup new -volatile]

        #
        # setup an uniqe drive and jukebox name
        #
        set nr 0
        while { $nr < 1000 } {
            incr nr
            set drivename1  "$driveprefix${nr}.1"
            set drivename2  "$driveprefix${nr}.2"
            set drivetext1  "$drivetextprefix1${nr}"
            set drivetext2  "$drivetextprefix2${nr}"
            set jukeboxname "$jukeboxprefix${nr}"
            set jukeboxtext "$jukeboxtextprefix${nr}"
            if { [$devicegrp ls $drivename1] == "" && [$devicegrp ls $drivename2] == "" && [$jukeboxgrp ls $jukeboxname] == "" } {
                break
            }
        }

        #
        # check if directory exists
        #
        if {![utility::file isdirectory $baseDir]} {
           set baseDir   [utility::file normalize $baseDir]
           set parentDir [utility::file dirname $baseDir]
           if {![utility::file isdirectory $parentDir]} {
               throw NOSUCHDIR
           }
           utility::file mkdir $baseDir
        }

        #
        # check if path is already in use from an other drive
        #

        foreach drv [$devicegrp ls] {
            set drv [$devicegrp $drv]
            if { [$drv baseDir] == $baseDir } {
                set drvname [$drv name]
                throw PATHINUSE
            }
        }

        #
        # calculated number of slots and volume size
        #

        if { $sizegb > 0 } {
            set disksize $sizegb
            # change from GB to Byte
            set disksize [expr {wide($disksize) * wide(0x40000000)}]
        } else {
            set disksize [volume avail $baseDir]
            # change from KB to Byte
            set disksize [expr {wide($disksize) * wide(0x400)}]
        }

        # if disk size smaller than 10x2GB
        if { $disksize <= $lowerborder && $disksize >= $lowervollimit } {
            set nrofslots  [expr {wide($disksize) / wide($lowervollimit)}]
            set mediumSize $minvol
        # if disk size larger than 10x2GB and smaller than 10xmaxvol
        } elseif { $disksize > $lowerborder && $disksize <= $upperborder } {
            set nrofslots  $slots
            set mediumSize [expr {wide($disksize) / $nrofslots}]
            set mediumSize [expr {wide($mediumSize) / wide(0x40000000)}]
        # if disk size larger than 10xmaxvol
        } elseif { $disksize > $upperborder } {
            set nrofslots  [expr {wide($disksize) / wide($uppervollimit)}]
            set mediumSize $maxvol
        } else {
            throw NOTSUPPORTEDSIZ
        }

        # check if the total volume size is smaller than disk free size
        set availsize   [volume avail $baseDir]
        set availsize   [expr {wide($availsize) * wide(0x400)}]
        if { [expr {wide($mediumSize) * wide(0x40000000) * $nrofslots}] > $availsize } {
            set availsize   [expr {wide($availsize) / wide(0x40000000)}]
            set jukeboxSize [expr {wide($mediumSize) * $nrofslots }]
            throw VOLTOLARGE
        }

        # last check for volume size
        if { $mediumSize > $maxvol } {
           throw MAXFILESIZ
        }

        #
        # create the disk library and drives
        #

        # create drive 1
        if { [catch { set thisDrive [$devicegrp new $drivename1 \
                       mediaType "DISK" \
                       deviceType "DISK" \
                       baseDir $baseDir \
                       mediumSize $mediumSize \
                       osPath [file join [const deviceNodesDirectory] $drivename1]] } err] } {
            throw ERRCREATEDEV1
        }
        $thisDrive describe $drivetext1
        $thisDrive detachVolume

        # create drive 2
        if { [catch { set thisDrive [$devicegrp new $drivename2 \
                       mediaType "DISK" \
                       deviceType "DISK" \
                       baseDir $baseDir \
                       mediumSize $mediumSize \
                       osPath [file join [const deviceNodesDirectory] $drivename2]] } err] } {
            throw ERRCREATEDEV2
        }
        #
        $thisDrive describe $drivetext2
        $thisDrive detachVolume

        # create jukebox
        if {[catch {VJBox new $jukeboxname $nrofslots [list $drivename1 $drivename2] cooked} thisJukebox]} {
            throw ERRCREATEVTL
        }
        $thisJukebox close
        set thisJukebox [$jukeboxgrp $jukeboxname]
        $thisJukebox describe $jukeboxtext
        $thisJukebox calcSlotCnt

        # do a first inventory
        MedMgr mount_inventory -jukebox $jukeboxname

        # set message [fmtmsg $message "Creating a $nrofslots slot disk library with a volume size of ${mediumSize}GBytes each."]


        #
        # check and create pool
        #

        if { $usageSwitch == "Backup" || $usageSwitch == "Archive" || $usageSwitch == "pool" } {
            if { $usageSwitch == "Backup" } {
                set poolname  $backuppoolname
                set pooltext  $backuppooltext
            } elseif { $usageSwitch == "Archive" } {
                set poolname  $archivepoolname
                set pooltext  $archivepooltext
            }

            if { [lsearch -exact [$poolgrp ls] $poolname] == -1 } {
                if { [catch { set thisPool [$poolgrp new $poolname mediaType "DISK" usage $usageSwitch]  } err] } {
                    throw ERRCREATEPOOL
                }
                $thisPool describe $pooltext
            }

            #
            # label volumes
            #

            # create list of slots
            set slotlist [list]
            for {set nr 1} { $nr <= $nrofslots } { incr nr } {
                lappend slotlist $nr
            }
            # label volumes
            set job [JobScheduler new -volatile]
            set jobticket [$job command now "MedMgr label -pool $poolname -slots {$slotlist} -jukebox $jukeboxname -noerase -skiperrors" [translate "Labeling for pool $poolname"]]
            # set jukebox default pool
            $thisJukebox defaultPool $poolname
        }
    } catch {PATHINUSE} {
        set message [fmtmsg $message "The directory '$baseDir' is already in use from an other drive '$drvname'." error]
        throw PATHINUSE $message
    } catch {NOTSUPPORTEDSIZ} {
        set message [fmtmsg $message "A disk size smaller than 2GBytes in not supported." error]
        throw NOTSUPPORTEDSIZ $message
    } catch {VOLTOLARGE} {
        set message [fmtmsg $message "The required disk size of $jukeboxSize GBytes exeeds the available space of $availsize GBytes." error]
        throw VOLTOLARGE $message
    } catch {MAXFILESIZ} {
        set message [fmtmsg $message "Maximum file size is $maxvol GBytes."]
        throw MAXFILESIZ $message
    } catch {NOSUCHDIR} {
        set message [fmtmsg $message "The parent directory '$parentDir' does not exist." error]
        throw NOSUCHDIR $message
    } catch {ERRCREATEDEV1} {
        set message [fmtmsg $message "Could not create drive1 '$drivename1': $err" error]
        throw ERRCREATEDEV1 $message
    } catch {ERRCREATEDEV2} {
        set message [fmtmsg $message "Could not create drive2 '$drivename2': $err" error]
        catch {
            $devicegrp delete $drivename1
        }
        throw ERRCREATEDEV2 $message
    } catch {ERRCREATEPOOL} {
        set message [fmtmsg $message "Could not create pool '$poolname': $err" error]
        throw ERRCREATEPOOL $message
    } catch {ERRCREATEVTL} {
        set message [fmtmsg $message "Could not create library '$jukeboxname': $thisJukebox" error]
        catch {
            $devicegrp delete $drivename1
            $devicegrp delete $drivename2
        }
        throw ERRCREATEVTL $message
    } catch {ENOLICENSE} {
        set message [fmtmsg $message "No valid license to enable resource '$jukeboxname'." error]
        throw ENOLICENSE $message
    } catch {*} {
        set message [fmtmsg $message $trymsg errorinfo]
    } finally {
        catch { $job destroy }
    } trymsg

    return $jukeboxname
}

##############################################################################

proc createDiskCloudLib6.1 { args } {

    # createDiskCloudLib -checkonly -baseDir -usageSwitch -objStore -clonedPool

    # ns_log notice args:$args

    array set C $args

    #ns_log notice checkonly:$C(-checkonly)
    #ns_log notice baseDir:$C(-baseDir)
    #ns_log notice usageSwitch:$C(-usageSwitch)
    #ns_log notice objStore:$C(-objStore)
    #ns_log notice clonedPool:$C(-clonedPool)

    set message              ""
    set volChunkSize         268435456

    # set up the libary
    set slots                10
    set minvol               2
    # a requirement from the support to reduce this from 2048 to 1024
    set maxvol               1024
    set lowervollimit        [expr {wide(0x40000000) * $minvol}]
    set lowerborder          [expr {wide($lowervollimit) * $slots}]
    set uppervollimit        [expr {wide(0x40000000) * $maxvol}]
    set upperborder          [expr {wide($uppervollimit) * $slots}]
    #
    try {
        #
        set devicegrp  [DeviceGroup new -volatile]
        set jukeboxgrp [JukeboxGroup new -volatile]
        set poolgrp    [PoolGroup new -volatile]
        set cloudgrp   [ObjectStoreGroup new -volatile]

        set driveprefix       "drive_"
        set drivetextprefix1  "Drive 1 of disk library "
        set drivetextprefix2  "Drive 2 of disk library "
        set jukeboxprefix     "disk_lib_"

        if { [info exists C(-objStore)] && $C(-objStore) != "" && [$cloudgrp ls $C(-objStore)] != "" } {
            if { [lsearch -exact [$cloudgrp ls] $C(-objStore)] >= 0 } {
                if { [set objStoreprefix [[$cloudgrp $C(-objStore)] describe]] != "" } {
                    regsub -all {[ ]} $objStoreprefix "_" objStoreprefix
                    regsub -all {[^A-Za-z0-9\_-]} $objStoreprefix "" objStoreprefix
                } else {
                    set objStoreprefix $C(-objStore)]
                }
                set backuppoolname  "${objStoreprefix}_bck"
                set archivepoolname "${objStoreprefix}_arc"
            } else {
                set backuppoolname  "Cloud-Backup"
                set archivepoolname "Cloud-Archive"
            }
        } else {
            set backuppoolname  "Disk-Backup"
            set archivepoolname "Disk-Archive"
        }

        if { $C(-usageSwitch) == "Backup" } {
            set poolprefix $backuppoolname
        } elseif { $C(-usageSwitch) == "Archive" } {
            set poolprefix $archivepoolname
        }

        #
        # setup an uniqe drive, jukebox and pool name
        #
        set nr 0
        while { $nr < 1000 } {
            incr nr
            set drivename1  "$driveprefix${nr}.1"
            set drivename2  "$driveprefix${nr}.2"
            set drivetext1  "$drivetextprefix1${nr}"
            set drivetext2  "$drivetextprefix2${nr}"
            set jukeboxname "$jukeboxprefix${nr}"
            if { [$devicegrp ls $drivename1] == "" && [$devicegrp ls $drivename2] == "" && [$jukeboxgrp ls $jukeboxname] == "" } {
                break
            }
        }
        set jbnr $nr
        set nr 0
        set poolname "$poolprefix"
        while { $nr < 1000 } {
            incr nr
            if { [$poolgrp ls $poolname] == "" } {
                break
            } else {
                set poolname "${poolprefix}_${nr}"
            }
        }

        set jukeboxtext "Disk lib $jbnr ($poolname)"
        set pooltext    "Volumes managed by $jukeboxname"

        #
        # check if directory exists
        #
        if {![utility::file isdirectory $C(-baseDir)]} {
           set C(-baseDir)   [utility::file normalize $C(-baseDir)]
           set parentDir [utility::file dirname $C(-baseDir)]
           if {![utility::file isdirectory $parentDir]} {
               throw NOSUCHDIR
           }
           utility::file mkdir $C(-baseDir)
        }

        #
        # check if path is already in use from an other drive
        #

        foreach drv [$devicegrp ls] {
            set drv [$devicegrp $drv]
            if { [$drv baseDir] == $C(-baseDir) } {
                set drvname [$drv name]
                throw PATHINUSE
            }
        }


        if { $C(-clonedPool) } {
            #
            # calculate the number of slots and volume size depending from disk size
            #
            set disksize [volume avail $C(-baseDir)]
            # change from KB to Byte
            set disksize [expr {wide($disksize) * wide(0x400)}]

            # if disk size smaller than 10x2GB
            if { $disksize <= $lowerborder && $disksize >= $lowervollimit } {
                set nrofslots  [expr {wide($disksize) / wide($lowervollimit)}]
                set mediumSize $minvol
            # if disk size larger than 10x2GB and smaller than 10xmaxvol
            } elseif { $disksize > $lowerborder && $disksize <= $upperborder } {
                set nrofslots  $slots
                set mediumSize [expr {wide($disksize) / $nrofslots}]
                set mediumSize [expr {wide($mediumSize) / wide(0x40000000)}]
            # if disk size larger than 10xmaxvol
            } elseif { $disksize > $upperborder } {
                set nrofslots  [expr {wide($disksize) / wide($uppervollimit)}]
                set mediumSize $maxvol
            } else {
                throw NOTSUPPORTEDSIZ
            }
        } else {
            #
            # calculate the number of slots and volume size depending on licensing
            #
            set nrofslots  100
            set mediumSize $maxvol
        }

        # last check for volume size
        if { $mediumSize > $maxvol } {
            throw MAXFILESIZ
        }

        # in case this proc is called as parameter check
        if { $C(-checkonly) } { return }

        #
        # create the disk library and drives
        #

        # create drive 1
        if { [catch { set thisDrive [$devicegrp new $drivename1 \
                       mediaType "DISK" \
                       deviceType "DISK" \
                       baseDir $C(-baseDir) \
                       mediumSize $mediumSize \
                       osPath [file join [const deviceNodesDirectory] $drivename1]] } err] } {
            throw ERRCREATEDEV1
        }
        $thisDrive describe $drivetext1
        $thisDrive detachVolume

        # create drive 2
        if { [catch { set thisDrive [$devicegrp new $drivename2 \
                       mediaType "DISK" \
                       deviceType "DISK" \
                       baseDir $C(-baseDir) \
                       mediumSize $mediumSize \
                       osPath [file join [const deviceNodesDirectory] $drivename2]] } err] } {
            throw ERRCREATEDEV2
        }
        #
        $thisDrive describe $drivetext2
        $thisDrive detachVolume

        # create jukebox
        if {[catch {VJBox new $jukeboxname $nrofslots [list $drivename1 $drivename2] cooked} thisJukebox]} {
            throw ERRCREATEVTL
        }
        $thisJukebox close
        set thisJukebox [$jukeboxgrp $jukeboxname]
        $thisJukebox describe $jukeboxtext
        $thisJukebox calcSlotCnt

        # do a first inventory
        MedMgr mount_inventory -jukebox $jukeboxname

        # set message [fmtmsg $message "Creating a $nrofslots slot disk library with a volume size of ${mediumSize}GBytes each."]


        #
        # check and create pool
        #

        if { $C(-usageSwitch) == "Backup" || $C(-usageSwitch) == "Archive" } {
            if { [lsearch -exact [$poolgrp ls] $poolname] == -1 } {
                if { [catch { set thisPool [$poolgrp new $poolname mediaType "DISK" usage $C(-usageSwitch) objStore $C(-objStore) clonedPool $C(-clonedPool) volChunkSize $volChunkSize strictStreams 1 maxStreams 1 useUpToDrives 1]  } err] } {
                    throw ERRCREATEPOOL
                }
                $thisPool describe $pooltext
            }
        }

        #
        # label volumes
        #

        if { $C(-usageSwitch) == "Backup" || $C(-usageSwitch) == "Archive" } {
            # create list of slots
            set slotlist [list]
            for {set nr 1} { $nr <= $nrofslots } { incr nr } {
                lappend slotlist $nr
            }
            # label volumes
            set job [JobScheduler new -volatile]
            set jobticket [$job command now "MedMgr label -pool $poolname -slots {$slotlist} -jukebox $jukeboxname -noerase -skiperrors" [translate "Labeling for pool $poolname"]]
            # set jukebox default pool
            $thisJukebox defaultPool $poolname

        }
    } catch {PATHINUSE} {
        set message [fmtmsg $message "The directory '$C(-baseDir)' is already in use from an other drive '$drvname'." error]
        throw PATHINUSE $message
    } catch {NOTSUPPORTEDSIZ} {
        set message [fmtmsg $message "A disk size smaller than 2GBytes in not supported." error]
        throw NOTSUPPORTEDSIZ $message
    } catch {MAXFILESIZ} {
        set message [fmtmsg $message "Maximum file size is $maxvol GBytes."]
        throw MAXFILESIZ $message
    } catch {NOSUCHDIR} {
        set message [fmtmsg $message "The parent directory '$parentDir' does not exist." error]
        throw NOSUCHDIR $message
    } catch {ERRCREATEDEV1} {
        set message [fmtmsg $message "Could not create drive1 '$drivename1': $err" error]
        throw ERRCREATEDEV1 $message
    } catch {ERRCREATEDEV2} {
        set message [fmtmsg $message "Could not create drive2 '$drivename2': $err" error]
        catch {
            $devicegrp delete $drivename1
        }
        throw ERRCREATEDEV2 $message
    } catch {ERRCREATEPOOL} {
        set message [fmtmsg $message "Could not create pool '$poolname': $err" error]
        throw ERRCREATEPOOL $message
    } catch {ERRCREATEVTL} {
        set message [fmtmsg $message "Could not create library '$jukeboxname': $thisJukebox" error]
        catch {
            $devicegrp delete $drivename1
            $devicegrp delete $drivename2
        }
        throw ERRCREATEVTL $message
    } catch {ENOLICENSE} {
        set message [fmtmsg $message "No valid license to enable resource '$jukeboxname'." error]
        throw ENOLICENSE $message
    } catch {*} {
        set message [fmtmsg $message $trymsg errorinfo]
    } finally {
        catch { $job destroy }
    } trymsg

    return $poolname
}

##############################################################################

proc createDiskCloudLib { args } {

    # -checkonly -baseDir -usageSwitch -objStore -clonedPool

    # ns_log notice args:$args

    array set C $args

    #ns_log notice returnRsrc:$C(-returnRsrc)
    #ns_log notice poolname:$C(-poolname)
    #ns_log notice checkonly:$C(-checkonly)
    #ns_log notice baseDirAsIs:$C(-baseDirAsIs)
    #ns_log notice baseDir:$C(-baseDir)
    #ns_log notice usageSwitch:$C(-usageSwitch)
    #ns_log notice objStore:$C(-objStore)
    #ns_log notice clonedPool:$C(-clonedPool)

    set message        ""
    set volChunkSize   268435456
    set mediaBlockSize 65536
    set blockSize      65536
    set volumename     ""

    #
    try {
        #
        set devicegrp  [DeviceGroup new -volatile]
        set poolgrp    [PoolGroup new -volatile]
        set volumegrp  [VolumeGroup new -volatile]
        set cloudgrp   [ObjectStoreGroup new -volatile]

        set driveprefix  "drive_"

        if { [info exists C(-objStore)] && $C(-objStore) != "" && [$cloudgrp ls $C(-objStore)] != "" } {
            if { [lsearch -exact [$cloudgrp ls] $C(-objStore)] >= 0 } {
                if { [set objStoreprefix [[$cloudgrp $C(-objStore)] describe]] != "" } {
                    regsub -all {[ ]} $objStoreprefix "_" objStoreprefix
                    regsub -all {[^A-Za-z0-9\_-]} $objStoreprefix "" objStoreprefix
                } else {
                    set objStoreprefix $C(-objStore)]
                }
                set backuppoolname  "${objStoreprefix}_bck"
                set archivepoolname "${objStoreprefix}_arc"
            } else {
                set backuppoolname  "Cloud-Backup"
                set archivepoolname "Cloud-Archive"
            }
        } else {
            set backuppoolname  "Cloud-Backup"
            set archivepoolname "Cloud-Archive"
        }

        if { $C(-usageSwitch) == "Backup" } {
            set poolprefix $backuppoolname
        } elseif { $C(-usageSwitch) == "Archive" } {
            set poolprefix $archivepoolname
        } elseif { $C(-usageSwitch) == "ConfigRestore" } {
            set poolprefix "Created_for_config_restore"
            set C(-usageSwitch) "Backup"
        }

        #
        # setup an uniqe drive, jukebox and pool name
        #
        set nr 0
        while { $nr < 1000 } {
            incr nr
            set drivename  "$driveprefix${nr}.1"
            if { [$devicegrp ls $drivename] == ""  } {
                break
            }
        }
        set nr 0
        if { [info exists C(-poolname)] && $C(-poolname) ne {} } {
            set poolname   $C(-poolname)
            set poolprefix $C(-poolname)
        } else {
            set poolname "$poolprefix"
        }
        while { $nr < 1000 } {
            incr nr
            if { [$poolgrp ls $poolname] == "" } {
                break
            } else {
                set poolname "${poolprefix}_${nr}"
            }
        }

        #
        # check if directory exists
        #
        if {![utility::file isdirectory $C(-baseDir)]} {
           set C(-baseDir)   [utility::file normalize $C(-baseDir)]
           set parentDir [utility::file dirname $C(-baseDir)]
           if {![utility::file isdirectory $parentDir]} {
               throw NOSUCHDIR
           }
           utility::file mkdir $C(-baseDir)
        }

        # in case this proc is called as parameter check
        if { $C(-checkonly) } { return }


        # if started from wizard add poolname as target directory
        # because of too many exceptions where it is necessary to use baseDir as it is
        # if { ![info exists C(-baseDirAsIs)] || !$C(-baseDirAsIs) }
        if { 0 } {
            set targetDir $C(-baseDir)/$poolname
            if {[utility::file isdirectory $targetDir]} {
               throw DIREXIST
            }
            utility::file mkdir $targetDir
        } else {
            set targetDir $C(-baseDir)
        }

        #
        # check if path is already in use from an other volume
        #

        foreach vol [$volumegrp ls] {
            set vol [$volumegrp $vol]
            if { [$vol baseDir] == $targetDir } {
                set volname [$vol name]
                throw PATHINUSE
            }
        }

        if { $C(-usageSwitch) == "Backup" || $C(-usageSwitch) == "Archive" } {

            #
            # create the drive
            #
            set thisDrive  ""
            set thisPool   ""
            set thisVolume ""
            if { [catch { set thisDrive [$devicegrp new $drivename \
                           mediaType "CONTAINER" \
                           deviceType "CONTAINER" \
                           baseDir $targetDir ] } err] } {
                throw CLEANUP
            }

            #
            # check and create pool and volume
            #

            if { [lsearch -exact [$poolgrp ls] $poolname] == -1 } {
                if { [catch { set thisPool [$poolgrp new $poolname mediaType "CONTAINER" usage $C(-usageSwitch) objStore $C(-objStore) clonedPool $C(-clonedPool) volChunkSize $volChunkSize mediaBlockSize $mediaBlockSize strictStreams 1 maxStreams 1 useUpToDrives 1]  } err] } {
                    throw CLEANUP
                }
            }
            if { [llength [$volumegrp find pool $poolname]] == 0 } {
                if { [catch { set thisVolume [$volumegrp new {} pool $poolname mediaType "CONTAINER" usage $C(-usageSwitch) objStore $C(-objStore) blockSize $blockSize drive $drivename baseDir $targetDir] } err] } {
                    throw CLEANUP
                }
                set volumename [$thisVolume name]
                $thisVolume describe "Container volume $poolname"
                $thisPool   describe "Container volume $volumename"
                $thisDrive  describe "Drive for $poolname"
                # link drive to volume
                $thisDrive volume $volumename
                # link pool to volume
                $thisPool  volume $volumename
            }
        }
    } catch {CLEANUP} {
        set message "Could not create resources for the container volume. Cleanup started."
        ns_log error $message
        ns_log error $err
        if { $thisDrive ne "" } {
            $thisDrive volume 0
            ns_log notice "Volume referenze in drive reseted: $drivename"
        }
        if { $thisVolume ne "" } {
            $thisVolume drive ""
            ns_log notice "Drive referenze in volume reseted: $volumename"
        }
        if { $thisDrive ne "" } {
            $devicegrp delete $drivename 1
            ns_log notice "Drive deleted: $drivename"
        }
        if { $thisVolume ne "" } {
            $volumegrp delete $volumename
            ns_log notice "Volume deleted: $volumename"
        }
        if { $thisPool ne "" } {
            $poolgrp delete $poolname
            ns_log notice "Pool deleted: $poolname"
        }
    } catch {DIREXIST} {
        set message "The directory '$targetDir' exist already."
        throw DIREXIST $message
    } catch {PATHINUSE} {
        set message "The directory '$targetDir' is already in use from an other volume '$volname'."
        throw PATHINUSE $message
    } catch {NOSUCHDIR} {
        set message "The parent directory '$parentDir' does not exist."
        throw NOSUCHDIR $message
    } catch {*} {
        return -code [TCL_ERROR] $trymsg
    } finally {
    } trymsg

    if { [info exists C(-returnRsrc)] && $C(-returnRsrc) eq "Volume" } {
        return $volumename
    } else {
        return $poolname
    }

}

##############################################################################

proc createContainerResources { args } {

    # -checkonly -baseDir -usageSwitch -objStore -clonedPool

    # ns_log notice args:$args

    array set C $args

    #ns_log notice returnRsrc:$C(-returnRsrc)
    #ns_log notice poolname:$C(-poolname)
    #ns_log notice baseDirAsIs:$C(-baseDirAsIs)
    #ns_log notice checkonly:$C(-checkonly)
    #ns_log notice baseDir:$C(-baseDir)
    #ns_log notice usageSwitch:$C(-usageSwitch)

    set message         ""
    set volChunkSize   268435456
    set mediaBlockSize 65536
    set blockSize      65536
    set volumename     ""

    #
    try {
        #
        set devicegrp  [DeviceGroup new -volatile]
        set poolgrp    [PoolGroup new -volatile]
        set volumegrp  [VolumeGroup new -volatile]

        set driveprefix      "drive_"

        set backuppoolname  "Local_disk_bck"
        set archivepoolname "Local_disk_arc"


        if { $C(-usageSwitch) == "Backup" } {
            set poolprefix $backuppoolname
        } elseif { $C(-usageSwitch) == "Archive" } {
            set poolprefix $archivepoolname
        } elseif { $C(-usageSwitch) == "ConfigRestore" } {
            set poolprefix "Created_for_config_restore"
            set C(-usageSwitch) "Backup"
        }

        #
        # setup an uniqe drive, jukebox and pool name
        #
        set nr 0
        while { $nr < 1000 } {
            incr nr
            set drivename  "$driveprefix${nr}.1"
            if { [$devicegrp ls $drivename] == ""  } {
                break
            }
        }
        set nr 0
        if { [info exists C(-poolname)] && $C(-poolname) ne {} } {
            set poolname   $C(-poolname)
            set poolprefix $C(-poolname)
        } else {
            set poolname "$poolprefix"
        }
        while { $nr < 1000 } {
            incr nr
            if { [$poolgrp ls $poolname] == "" } {
                break
            } else {
                set poolname "${poolprefix}_${nr}"
            }
        }

        #
        # check if directory exists
        #
        if {![utility::file isdirectory $C(-baseDir)]} {
           set C(-baseDir)   [utility::file normalize $C(-baseDir)]
           set parentDir [utility::file dirname $C(-baseDir)]
           if {![utility::file isdirectory $parentDir]} {
               throw NOSUCHDIR
           }
           utility::file mkdir $C(-baseDir)
        }

        # in case this proc is called as parameter check
        if { $C(-checkonly) } { return }

        # if started from wizard add poolname as target directory
        # because of too many exceptions where it is necessary to use baseDir as it is
        # if { ![info exists C(-baseDirAsIs)] || !$C(-baseDirAsIs) }
        if { 0 } {
            set targetDir $C(-baseDir)/$poolname
            if {[utility::file isdirectory $targetDir]} {
               throw DIREXIST
            }
            utility::file mkdir $targetDir
        } else {
            set targetDir $C(-baseDir)
        }

        #
        # check if path is already in use from an other volume
        #

        foreach vol [$volumegrp ls] {
            set vol [$volumegrp $vol]
            if { [$vol baseDir] == $targetDir } {
                set volname [$vol name]
                throw PATHINUSE
            }
        }

        if { $C(-usageSwitch) == "Backup" || $C(-usageSwitch) == "Archive" } {

            #
            # create the drive
            #
            set thisDrive  ""
            set thisPool   ""
            set thisVolume ""
            if { [catch { set thisDrive [$devicegrp new $drivename \
                           mediaType "CONTAINER" \
                           deviceType "CONTAINER" \
                           baseDir $targetDir ] } err] } {
                throw CLEANUP
            }

            #
            # check and create pool and volume
            #

            if { [lsearch -exact [$poolgrp ls] $poolname] == -1 } {
                if { [catch { set thisPool [$poolgrp new $poolname mediaType "CONTAINER" usage $C(-usageSwitch) volChunkSize $volChunkSize mediaBlockSize $mediaBlockSize strictStreams 1 maxStreams 1 useUpToDrives 1]  } err] } {
                    throw CLEANUP
                }
            }
            if { [llength [$volumegrp find pool $poolname]] == 0 } {
                if { [catch { set thisVolume [$volumegrp new {} pool $poolname mediaType "CONTAINER" usage $C(-usageSwitch) blockSize $blockSize drive $drivename baseDir $targetDir] } err] } {
                    throw CLEANUP
                }
                set volumename [$thisVolume name]
                $thisVolume describe "Container volume $poolname"
                $thisPool   describe "Container volume $volumename"
                $thisDrive  describe "Drive for $poolname"
                # link drive to volume
                $thisDrive volume $volumename
                # link pool to volume
                $thisPool  volume $volumename
            }
        }
    } catch {CLEANUP} {
        set message "Could not create resources for the container volume. Cleanup started."
        ns_log error $message
        ns_log error $err
        if { $thisDrive ne "" } {
            $thisDrive volume 0
            ns_log notice "Volume referenze in drive reseted: $drivename"
        }
        if { $thisVolume ne "" } {
            $thisVolume drive ""
            ns_log notice "Drive referenze in volume reseted: $volumename"
        }
        if { $thisDrive ne "" } {
            $devicegrp delete $drivename 1
            ns_log notice "Drive deleted: $drivename"
        }
        if { $thisVolume ne "" } {
            $volumegrp delete $volumename
            ns_log notice "Volume deleted: $volumename"
        }
        if { $thisPool ne "" } {
            $poolgrp delete $poolname
            ns_log notice "Pool deleted: $poolname"
        }
    } catch {DIREXIST} {
        set message "The directory '$targetDir' exist already."
        throw DIREXIST $message
    } catch {PATHINUSE} {
        set message "The directory '$targetDir' is already in use from an other volume '$volname'."
        throw PATHINUSE $message
    } catch {NOSUCHDIR} {
        set message "The parent directory '$parentDir' does not exist."
        throw NOSUCHDIR $message
    } catch {*} {
        return -code [TCL_ERROR] $trymsg
    } finally {
    } trymsg

    if { [info exists C(-returnRsrc)] && $C(-returnRsrc) eq "Volume" } {
        return $volumename
    } else {
        return $poolname
    }

}

##############################################################################

proc writeLTFS { srchost srcpath tgthost tgtpath volroot} {


    ns_log notice "LTFS write from '$srchost:$srcpath' to '$tgthost:$tgtpath'"

    set returnCode [catch {JobMgr syncdirs $srchost $srcpath $tgthost $tgtpath} result]

    if { $returnCode == [TCL_OK] || $returnCode == [TCL_WARNING] } {
        [JobScheduler new -volatile] command now "MedMgr sync_tape_index -mountpoint $volroot" "File system sync (index flush) triggered for $volroot"
    }

    return -code $returnCode $result

}

##############################################################################

proc readLTFS { srchost srcpath tgthost tgtpath } {

    ns_log notice "LTFS read from '$srchost:$srcpath' to '$tgthost:$tgtpath'"

    set result [JobMgr syncdirs $srchost $srcpath $tgthost $tgtpath {} -sort +ino]

    return $result

}

##############################################################################

proc restoreLTFS { relocateClient selectHandle bixtreePath resolution rtimeFlag} {


    ns_log notice "Restore to LTFS '$bixtreePath'"

    set returnCode [catch {JobMgr restore $relocateClient $selectHandle -relocatepath $bixtreePath -resolution $resolution -setrtime $rtimeFlag} result]

    if { $returnCode == [TCL_OK] || $returnCode == [TCL_WARNING] } {

        set LFile   [LexxFile new -volatile]
        set volroot [$LFile volroot $bixtreePath]

        [JobScheduler new -volatile] command now "MedMgr sync_tape_index -mountpoint $volroot" "File system sync (index flush) triggered for $volroot"

    }

    return -code $returnCode $result

}

##############################################################################

proc manualSync { srchost srcdirs tgthost tgtroot } {

    set rcode   0
    set result  ""
    set listing ""
    set wrnFlag 0

    try {

        foreach client [list $srchost $tgthost] {
            switch -exact -- [Agent ping $client 5] {
                -4 { throw CLIENT-4 }
                -3 { throw CLIENT-3 }
                -2 { throw CLIENT-2 }
                -1 { throw CLIENT-1 }
            }
        }

        if { $srchost == "localhost" } {
            set srcFileObj [LexxFile new -volatile]
        } else {
            set srcAgent   [Agent connect $srchost]
            set srcFileObj [$srcAgent remote LexxFile]
        }

        if { $tgthost == "localhost" } {
            set tgtFileObj [LexxFile new -volatile]
        } else {
            set tgtAgent   [Agent connect $tgthost]
            set tgtFileObj [$tgtAgent remote LexxFile]
        }

        foreach path $srcdirs {
            # check if source exists
            if { ![set rcode [catch {$srcFileObj lstat $path buf} result]] } {
                # check if source is a directory
                if { [$srcFileObj isdirectory $path] } {
                    # create target directory
                    # using tcl file because tail and path are not implemeted
                    set tgtdir  [file tail $path]
                    set tgtpath [file join $tgtroot $tgtdir]
                    if { [set rcode [catch {$tgtFileObj mkdir $tgtpath} result]] } {
                        throw ERRMKDIR
                    }
                } else {
                    set tgtpath $tgtroot
                }
            } else {
                throw NOSRCPATH
            }

            # use same syncdirs call for directories and files
            ns_log notice "Manual Sync from '$srchost:$path' to '$tgthost:$tgtpath'"
            set rcode [catch {JobMgr syncdirs $srchost $path $tgthost $tgtpath {} -overwrite 0} result]
            if { $rcode == [TCL_ERROR] || $rcode == [TCL_BREAK] } {
                throw ERRSYNC
            } else {
                append listing $result \n
                if { $rcode == [TCL_WARNING] } {
                    set wrnFlag 1
                }
            }
        }

        # if no error set return result to the list of sync results
        set result $listing

        # incase target directory is on LTFS
        if { $tgthost == "localhost" && [regexp "^/ltfs" $tgtpath] && ( $rcode == [TCL_OK] || $rcode == [TCL_WARNING] ) } {
            set volroot [$tgtFileObj volroot $tgtpath]
            [JobScheduler new -volatile] command now "MedMgr sync_tape_index -mountpoint $volroot" "File system sync (index flush) triggered for $volroot"
        }

    } catch {ERRSYNC} {
        set result "[translate "Could not manual sync to folder: '$tgtpath'"]"
    } catch {ERRMKDIR} {
        set result "[translate "Could not create folder: '$tgtpath'"]"
    } catch {NOSRCPATH} {
        set result "[translate "No such file or directory: '$path'"]"
    } catch {CLIENT-4} {
        set rcode  [TCL_ERROR]
        set result "[translate "Could not connect to client '$client':"] [translate "unsupported software version"]"
    } catch {CLIENT-3} {
        set rcode   [TCL_ERROR]
        set result "[translate "Could not connect to client '$client':"] [translate "client disabled"]"
    } catch {CLIENT-2} {
        set rcode   [TCL_ERROR]
        set result "[translate "Could not connect to client '$client':"] [translate "wrong username/password"]"
    } catch {CLIENT-1} {
        set rcode   [TCL_ERROR]
        set result "[translate "Could not connect to client '$client':"] [translate "network connection problem"]"
    } catch {*} {
        set rcode   [TCL_ERROR]
        set result  [translate "Could not start manual sync."]
        ns_log error "manualSync: $::errorInfo"
    } finally {
        catch { $srcAgent destroy }
        catch { $tgtAgent destroy }
    } trymsg

    if { $wrnFlag } { set rcode [TCL_WARNING] }

    return -code $rcode $result

}

##############################################################################


